/**********************************************************************

	ファイルパス設定処理                          種別：sub

    (1)処理概要
	    パラメータで指定したファイルパスを記憶する。

    (2)呼出形式
			int setTimeCalFile( const char *filepath );
			int setLeapSecFile( const char *filepath );

    (3)パラメタ説明
	    ・ filepath (入力／char)
		   記憶するファイルますを指定する。（MAX128byte）

	    ・ 復帰値 (出力／int)
	    		RET_NORMAL				： 正常終了
				RET_PARAM_ERR			： 入力パラメータエラー
				RET_FILE_NOTFOUND_ERR	： 指定ファイルなし

    (4) 作成日 2012/11/22  Ver2.0 y.yoneda  新規作成
               2013/08/    Ver3.0.0 fips
               2023/07/31  Ver3.5.0 FJT     filepath_mng()呼出時の
											第2引数型キャスト

***********************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<errno.h>
#include	<string.h>
#include	<unistd.h>
#include	<inttypes.h>
#include	"TimeCal_type.h"	/* Chg Ver2.0 */
#include	"SATCA_com.h"

/*****************************************
	ファイルパス管理

	パラメータ
		flg		 :処理指定
					衛星時刻校正表ファイルパス設定	：TMCAL_FILE_PATH_SET(1)
					衛星時刻校正表ファイルパス取得	：TMCAL_FILE_PATH_GET(2)
					うるう秒ファイルパス設定		：LEAP_FILE_PATH_SET(3)
					うるう秒ファイルパス取得		：LEAP_FILE_PATH_GET(4)
		filepath :ファイルパス（フルパス）

	return
		RET_NORMAL			：正常終了
							：設置時：ファイルパス設定正常
							：抽出時：読み込み済みファイルパス
		RET_FILE_INIT_READ	：ファイル初期（再）読み込み必要
		RET_FILE_UNSET_ERR	：ファイルパス未設定

******************************************/
int filepath_mng( int flg, char *filepath )
{
	static	char	TimeCalFilePath[FILE_PATH_MAX_LENGTH+1];
	static	char	LeapSecFilePath[FILE_PATH_MAX_LENGTH+1];
	static	char	ErrMsgFilePath[FILE_PATH_MAX_LENGTH+1];		/* ver3.0.0 add */
	static	int		TimeCalFileInit = -1;
	static	int		LeapSecFileInit = -1;
	static	int		ErrMsgFileInit = -1;						/* ver3.0.0 add */
	int r = RET_NORMAL;

	if( flg == TMCAL_FILE_PATH_SET ) {
		strcpy( TimeCalFilePath, filepath );
		TimeCalFileInit = FILE_INIT_READ;		/* ファイル再読み込み要設定 */
	}
	else if( flg == TMCAL_FILE_PATH_GET ) {
		if( TimeCalFileInit == -1 ) {
			return(RET_FILE_UNSET_ERR);			/* ファイルパス未設定 */
		}
		strcpy( filepath, TimeCalFilePath );
		r = TimeCalFileInit;
		TimeCalFileInit = 0;
	}
	else if( flg == LEAP_FILE_PATH_SET ) {
		strcpy( LeapSecFilePath, filepath );
		LeapSecFileInit = FILE_INIT_READ;		/* ファイル再読み込み要設定 */
	}
	else if( flg == LEAP_FILE_PATH_GET ) {
		if( LeapSecFileInit == -1 ) {
			return(RET_FILE_UNSET_ERR);			/* ファイルパス未設定 */
		}
		strcpy( filepath, LeapSecFilePath );
		r = LeapSecFileInit;
		LeapSecFileInit = 0;
	}
	/* ver3.0.0 add エラーメッセージ出力ファイル */
	else if( flg == ERROR_FILE_PATH_SET ) {
		strcpy( ErrMsgFilePath, filepath );
		ErrMsgFileInit = 0;		/*  */
	}
	else if( flg == ERROR_FILE_PATH_GET ) {
		if( ErrMsgFileInit == -1 ) {
			strcpy( filepath, "/dev/stderr" );	/* 未設定の場合は標準エラー */
		}else{
			strcpy( filepath, ErrMsgFilePath );
		}
	}

	return(r);
}

/* 衛星時刻校正表のファイルパス設定 */
int setTimeCalFile( const char *filepath )
{
	/* 入力パラメータチェック */
	if( filepath == NULL || strlen(filepath) > (size_t)FILE_PATH_MAX_LENGTH ) {
		return(RET_PARAM_ERR);
	}

	/* ファイルの存在確認 */
	if( access( filepath, F_OK ) != 0 ) {
		return(RET_FILE_NOTFOUND_ERR);
	}

	/* ファイルパス登録 */
	filepath_mng( TMCAL_FILE_PATH_SET, (char *)filepath ); /* MOD 2023.07.31 CAST */

	/* ファイル再読み込み指定*/

	return(RET_NORMAL);
}

/* うるう秒ファイルのファイルパス設定 */
int setLeapSecFile( const char *filepath )
{
	/* 入力パラメータチェック */
	if( filepath == NULL || strlen(filepath) > (size_t)FILE_PATH_MAX_LENGTH ) {
		return(RET_PARAM_ERR);
	}

	/* ファイルの存在確認 */
	if( access( filepath, F_OK ) != 0 ) {
		return(RET_FILE_NOTFOUND_ERR);
	}

	/* ファイルパス登録 */
	filepath_mng( LEAP_FILE_PATH_SET, (char *)filepath ); /* MOD 2023.07.31 CAST */
	/* ファイル再読み込み指定*/

	return(RET_NORMAL);
}

/* エラーメッセージ出力ファイルのファイルパス設定 ver3.0.0 create */
int setErrorMsgFile( const char *filepath )
{
	/* 入力パラメータチェック */
	if( filepath == NULL || strlen(filepath) > (size_t)FILE_PATH_MAX_LENGTH ) {
		return(RET_PARAM_ERR);
	}

	/* ファイルパス登録 */
	filepath_mng( ERROR_FILE_PATH_SET, (char *)filepath ); /* MOD 2023.07.31 CAST */

	return(RET_NORMAL);
}

