/**********************************************************************

	ETI＝＞経過秒変換処理	                          種別：sub

    (1)処理概要
	    パラメータで指定したETIを衛星時刻に変換して通知する。

    (2)呼出形式
            int eti2es ( uint32_t eti_ext, uint32_t eti_base, double *elapsed_seconds )

    (3)パラメタ説明
	    ・ eti_ext (入力／uint32_t)
		   変換するETIを指定する。(単位：サイクル数)

	    ・ eti_base (入力／uint32_t)
		   変換する時刻 (TI) を指定する。値は31.25msec(2^-5sec)

	    ・ elapsed_seconds (出力／double)
		   変換した衛星時刻が格納される。(単位：秒)

	    ・ 復帰値 (出力／int)
				RET_NORMAL					：正常終了
				RET_BEFORE_FIRST_RECORD		：校正結果が衛星時刻校正表の最初のレコード以前（最初のレコードからの外挿）
				RET_AFTER_LAST_RECORD		：校正結果が衛星時刻校正表の最終レコード以降（最終レコードからの外挿）
				RET_BETWEEN_DISCONT_RECORDS	：校正結果がTI不連続区間に該当（外挿した）
				RET_INAPPROPRIATE_PART		：指定etiの校正結果が別の連続区間にある
				RET_PARAM_ERR				：入力パラメータチェックエラー
				RET_FILE_UNSET_ERR			：ファイルパス未設定
				RET_FILE_OPEN_ERR			：ファイルオープンエラー
				RET_FILE_READ_ERR			：ファイルリードエラー
				RET_EXT_NOTFOUND_ERR		：指定ext無し
				RET_CALC_ERR				：算出不可

    (4) 作成日 2011/02/10
            2012/11/15  Ver2.0 y.yoneda  インデントの調整
			2012/11/21  Ver2.0 y.yoneda  Ver2.0 対応
			2013/08/    Ver3.0.0 fips    校正表のペアを検索する際、ETI拡張部とETI基本部を
			                             別々に計算するように修正
			2015/12/22  Ver3.1.2 fae     連続区間でのロールオーバー検出時は過去側ではなく
			                             未来側から有効レートを探すように修正

***********************************************************************/

#include	<stdio.h>
#include	<errno.h>
#include	<string.h>
#include	<stdlib.h>
#include	<inttypes.h>
#include	"TimeCal_type.h"	/* Chg Ver2.0 */
#include	"SATCA_com.h"

int eti2es ( uint32_t eti_ext, uint32_t eti_base, double *elapsed_seconds )
{

	/****************************************************************/
	/* 領域定義 													*/
	/****************************************************************/
	static	TiTm_tbl	titm;	/* 基準時刻設定領域 */

	int		i, ir;
	double	ETI_TI;

	/* 2013/08/  ver3.0.0 Add */
	double	base_t;		/* ETI基本部の時刻計算用 拡張部(ext_t)と分けて計算するため */
	double	ext_t;		/* ETI拡張部の時刻計算用 */
	int		iret = RET_NORMAL;

	/* Add Ver2.0 パラメータチェック */
	if( elapsed_seconds == NULL ) {
		return(RET_PARAM_ERR);
	}

	/***********************************************************/
	/*   基準ファイル設定処理　　　　　　　                    */
	/***********************************************************/
	ir = TiTm_Init_Eti( &titm, eti_ext, eti_base );	/* Chg Ver2.0 */
	if( ir != RET_NORMAL){					/* Chg Ver2.0 */
#ifdef DBG_TITM
		printf("eti2es[%d] TiTm_Init_Eti Error!! return-code[%d] [%d] [%d]\n",__LINE__, ir, eti_ext, eti_base );
#endif
		return(ir);
	}

	if(titm.SATCA_cnv_rdcnt == 0){
#ifdef DBG_TITM
		printf("eti2es[%d] SATCA_cnv_rdcnt = 0  [%d] \n", __LINE__, titm.SATCA_cnv_rdcnt );
#endif
		return(RET_CALC_ERR);	/* Chg Ver2.0 */
	}

	/****************************************************************/
	/* TIからUTへの変換処理											*/
	/****************************************************************/
	/* Add&Chg Ver2.0 Start 以降の処理は全体的に変更 */
	double rate = 0.0;
	int readrec = 0;


	/* 2013/08/  ver3.0.0 Chg start */

	/* 先頭レコードから 指定ETI拡張部が先頭レコードのよりも前 */
	if( eti_ext < titm.SATCA_cnv_tbl[0].ETI ){
		/* 有効レート探す */
		rate = find_valid_rate(0, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		if( (double)eti_base > titm.SATCA_cnv_tbl[0].tpktTI ){
			/* Chg Ver3.1.2 xffmax -> xffmax+1.0 */
			base_t = ((titm.SATCA_cnv_tbl[0].tpktTI + titm.SATCA_Init.xffmax+1.0) - (double)eti_base) * rate;
			ext_t = ((titm.SATCA_cnv_tbl[0].ETI - (eti_ext+1)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}else{
			base_t = (titm.SATCA_cnv_tbl[0].tpktTI - (double)eti_base) * rate;
			ext_t = ((titm.SATCA_cnv_tbl[0].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}
		*elapsed_seconds = titm.SATCA_cnv_tbl[0].tpktTI_SATsec - ext_t - base_t;

		if( titm.SATCA_cnv_topflg == 1 ) {
			/* 校正表の先頭レコードより前 */
			return(RET_BEFORE_FIRST_RECORD);
		}
	/* 先頭レコードから 指定ETI拡張部が先頭レコードと同じ、ETI基本部が先頭レコードよりも前 */
	}else if( eti_ext == titm.SATCA_cnv_tbl[0].ETI && (double)eti_base < titm.SATCA_cnv_tbl[0].tpktTI ){
		/* 有効レート探す */
		rate = find_valid_rate(0, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		base_t = (titm.SATCA_cnv_tbl[0].tpktTI - (double)eti_base) * rate;
		*elapsed_seconds = titm.SATCA_cnv_tbl[0].tpktTI_SATsec - base_t;

		if( titm.SATCA_cnv_topflg == 1 ) {
			/* 校正表の先頭レコードより前 */
			return(RET_BEFORE_FIRST_RECORD);
		}
	/* 最終レコードから */
	}else if( eti_ext > titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI ){
		/* 有効レート探す */
		rate = find_valid_rate(titm.SATCA_cnv_rdcnt-1, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		if( (double)eti_base < titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI ){
			/* Chg Ver3.1.2 xffmax -> xffmax+1.0 */
			base_t = (((double)eti_base + titm.SATCA_Init.xffmax+1.0) - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI) * rate;
			ext_t = ((eti_ext-1 - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI ) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}else{
			base_t = ((double)eti_base - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI) * rate;
			ext_t = ((eti_ext - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}
		*elapsed_seconds = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec + ext_t + base_t;

		if( titm.SATCA_cnv_endflg == 1 ) {
			/* 最終レコード+1よりも離れている場合はエラー */
			if( eti_ext > titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI+1 ){
				// error
				return(RET_EXT_NOTFOUND_ERR);
			}
			return(RET_AFTER_LAST_RECORD);
		}
	/* 最終レコードから */
	}else if( (eti_ext == titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI) && (titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI < (double)eti_base)  ){
		/* 有効レート探す */
		rate = find_valid_rate(titm.SATCA_cnv_rdcnt-1, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		base_t = ((double)eti_base - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI) * rate;
		*elapsed_seconds = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec + base_t;

		if( titm.SATCA_cnv_endflg == 1 ) {
			/* 校正表の先頭レコードより前 */
			return(RET_AFTER_LAST_RECORD);
		}
	}else{
		/* 読み込み済みの校正表レコード数分Loop */
		for( i=0; i<titm.SATCA_cnv_rdcnt-1; i++ ) {
			/* 過去レコード(i)とETI拡張部、ETI基本部が一致 */
			if( titm.SATCA_cnv_tbl[i].ETI == eti_ext && titm.SATCA_cnv_tbl[i].tpktTI == (double)eti_base ){
				*elapsed_seconds = titm.SATCA_cnv_tbl[i].tpktTI_SATsec;
				break;
			/* 未来レコード(i+1)とETI拡張部、ETI基本部が一致 */
			}else if( titm.SATCA_cnv_tbl[i+1].ETI == eti_ext && titm.SATCA_cnv_tbl[i+1].tpktTI == (double)eti_base ){
				*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec;
				break;
			}else if( titm.SATCA_cnv_tbl[i].ETI <= eti_ext && eti_ext <= titm.SATCA_cnv_tbl[i+1].ETI ){
				if( titm.SATCA_cnv_tbl[i].ETI == titm.SATCA_cnv_tbl[i+1].ETI ){
					if( titm.SATCA_cnv_tbl[i].tpktTI <= (double)eti_base && (double)eti_base <= titm.SATCA_cnv_tbl[i+1].tpktTI ){
						/* 未来レコード(i+1)から計算 */
						/* 有効レート探す */
						rate = find_valid_rate(i+1, &titm);
						if( rate <= 0.0 ) {
							return(RET_CALC_ERR);
						}

						/* ETI拡張部、ETI基本部を別々に計算し、差分の時間を求める */
						base_t = (titm.SATCA_cnv_tbl[i+1].tpktTI - (double)eti_base) * rate;
						ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;

						*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - ext_t - base_t;
						break;
					}else{
						continue;
					}
				}else{	/* if(titm.SATCA_cnv_tbl[i].ETI < titm.SATCA_cnv_tbl[i+1].ETI ) */
					/* Add Ver3.1.2 連続区間でのロールオーバー判定追加 */
					if( titm.SATCA_cnv_tbl[i].ETI == eti_ext && strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ){
						/* 過去レコード(i)から計算 */
						/* 有効レート探す */
						rate = find_valid_rate(i, &titm);
						if( rate <= 0.0 ) {
							return(RET_CALC_ERR);
						}

						/* ETI拡張部、ETI基本部を別々に計算し、差分の時間を求める */
						base_t = ((double)eti_base - titm.SATCA_cnv_tbl[i].tpktTI) * rate;
						ext_t = ((eti_ext - titm.SATCA_cnv_tbl[i].ETI) * (titm.SATCA_Init.xffmax+1.0)) * rate;

						*elapsed_seconds = titm.SATCA_cnv_tbl[i].tpktTI_SATsec + ext_t + base_t;

						/* 不連続区間の場合 */
						if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ) {
							/* 校正結果がTI-UTCペアの間にある */
							if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < *elapsed_seconds) && (*elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ){
								return(RET_BETWEEN_DISCONT_RECORDS);
							}else{
								return(RET_INAPPROPRIATE_PART);
							}
						}
						break;
					}else if( titm.SATCA_cnv_tbl[i+1].ETI == eti_ext ){
						if( (double)eti_base < titm.SATCA_cnv_tbl[i+1].tpktTI ){
							/* 未来レコード(i+1)から計算 */
							/* 有効レート探す */
							rate = find_valid_rate(i+1, &titm);
							if( rate <= 0.0 ) {
								return(RET_CALC_ERR);
							}

							/* ETI拡張部、ETI基本部を別々に計算し、差分の時間を求める */
							base_t = (titm.SATCA_cnv_tbl[i+1].tpktTI - (double)eti_base) * rate;
							ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;

							*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - ext_t - base_t;

							/* 不連続区間の場合 */
							if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ) {
								/* 校正結果がTI-UTCペアの間にある */
								if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < *elapsed_seconds) && (*elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ){
									return(RET_BETWEEN_DISCONT_RECORDS);
								}else{
									return(RET_INAPPROPRIATE_PART);
								}
							}
							break;
						}else{
							continue;
						}
					}else{
						/* 未来レコード(i+1)から計算 */
						/* 有効レート探す */
						rate = find_valid_rate(i+1, &titm);
						if( rate <= 0.0 ) {
							return(RET_CALC_ERR);
						}

						if( (double)eti_base > titm.SATCA_cnv_tbl[i+1].tpktTI ){
							/* Chg Ver3.1.2 xffmax -> xffmax+1.0 */
							base_t = ((titm.SATCA_cnv_tbl[i+1].tpktTI + titm.SATCA_Init.xffmax+1.0) - (double)eti_base) * rate;
							ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext+1)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
						}else{
							base_t = (titm.SATCA_cnv_tbl[i+1].tpktTI - (double)eti_base) * rate;
							ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
						}
						*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - ext_t - base_t;

						/* 不連続区間の場合 */
						if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ) {
							/* 校正結果がTI-UTCペアの間にある */
							if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < *elapsed_seconds) && (*elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ){
								return(RET_BETWEEN_DISCONT_RECORDS);
							}else{
								return(RET_INAPPROPRIATE_PART);
							}
						}
						break;
					}
				}
			}
		}
	}


	/* 2013/08/  ver3.0.0 Chg end */


	return( RET_NORMAL );
	/* Add&Chg Ver2.0  End  */

}


/****************************************************************/
/* レート検索 										*/
/* 	read_cnt	：検索の基準となるレコード	 					*/
/* 	titm		：時刻校正用変換テーブルポインタ 				*/
/*																*/
/*	2013/08/ ver3.0.0 add										*/
/****************************************************************/
double find_valid_rate(int read_cnt, TiTm_tbl *titm)
{
	int raterec = 0;
	double ret = 0.0;
	int bef_rec = -1;
	double bef_interval = -1.0;
	int aft_rec = -1;
	double aft_interval = -1.0;

#ifdef DBG_TITM
	printf( "eti2es[%d] <find_valid_rate> \n", __LINE__ );
#endif
	/* 指定レコードのE/NフラグがEでなければ、そのレコードのレートを返す */
	if( titm->SATCA_cnv_tbl[read_cnt].quality_flag[0] != 'E' ) {
		return titm->SATCA_cnv_tbl[read_cnt].tpktTI_dT;
	}else{

		int rec_cnt = 1;
		while( 1 ) {
			/* 過去方向の有効なレートが見つかっていなければ探す */
			if( ((read_cnt - rec_cnt) >= 0) && bef_rec == -1 ) {
				/* 基準のレコードとの時間差を取得 */
				bef_interval = titm->SATCA_cnv_tbl[read_cnt].tpktTI_SATsec - titm->SATCA_cnv_tbl[read_cnt - rec_cnt].tpktTI_SATsec;
				if( titm->SATCA_cnv_tbl[read_cnt - rec_cnt].quality_flag[0] != 'E' ) {
					bef_rec = read_cnt - rec_cnt;
					/* 未来のレコードとの時間差を取得済みなら、時間差を比較して過去が基準レコードに近ければLoop抜ける */
					if( (aft_interval != -1.0 && (bef_interval < aft_interval)) ||
							((read_cnt + rec_cnt) >= titm->SATCA_cnv_rdcnt) ){
						break;
					}
				}
			}
			/* 未来方向の有効なレートが見つかっていなければ探す */
			if( ((read_cnt + rec_cnt) < titm->SATCA_cnv_rdcnt) && aft_rec == -1 ) {
				aft_interval = titm->SATCA_cnv_tbl[read_cnt + rec_cnt].tpktTI_SATsec - titm->SATCA_cnv_tbl[read_cnt].tpktTI_SATsec;
				if( titm->SATCA_cnv_tbl[read_cnt + rec_cnt].quality_flag[0] != 'E' ) {
					/* 基準のレコードとの時間差を取得 */
					aft_rec = read_cnt + rec_cnt;
					/* 過去の有効なレートが見つかっていたら、時間差を比較して未来が基準レコードに近ければLoop抜ける
					 * または、過去方向に読み切っていたらLoop抜ける */
					if( (bef_interval != -1.0 && (aft_interval < bef_interval)) ||
							((read_cnt - rec_cnt) < 0) ){
						break;
					}
				}
			}

			if( (read_cnt - rec_cnt) < 0 && (read_cnt + rec_cnt) >= titm->SATCA_cnv_rdcnt ) {
				/* 校正表テーブルを読み切った */
				break;
			}
			rec_cnt++;
		}

		/* 過去・未来両方の有効なレートが見つかっていた場合 */
		if( bef_rec != -1 && aft_rec != -1 ){
			if( bef_interval < aft_interval ){
				return(titm->SATCA_cnv_tbl[bef_rec].tpktTI_dT);
			}else{
				return(titm->SATCA_cnv_tbl[aft_rec].tpktTI_dT);
			}
		/* 過去の有効レートのみ */
		}else if(bef_rec != -1){
			return(titm->SATCA_cnv_tbl[bef_rec].tpktTI_dT);
		/* 未来の有効レートのみ */
		}else if(aft_rec != -1){
			return(titm->SATCA_cnv_tbl[aft_rec].tpktTI_dT);
		/* 有効レート見つからず */
		}else{
			return(-1.0);
		}


/*
		while( 1 ) {
			if( ((read_cnt - rec_cnt) >= 0) && bef_interval == -1.0 ) {
				if( titm->SATCA_cnv_tbl[read_cnt - rec_cnt].quality_flag[0] != 'E' ) {
					bef_interval = titm->SATCA_cnv_tbl[read_cnt].tpktTI_SATsec - titm->SATCA_cnv_tbl[read_cnt - rec_cnt].tpktTI_SATsec;
					if( aft_interval != -1.0 ){

					}
				}
			}
			if( ((read_cnt + rec_cnt) < titm->SATCA_cnv_rdcnt) && aft_interval == -1.0 ) {
				if( titm->SATCA_cnv_tbl[read_cnt + rec_cnt].quality_flag[0] != 'E' ) {
					aft_interval = titm->SATCA_cnv_tbl[read_cnt + rec_cnt].tpktTI_SATsec - titm->SATCA_cnv_tbl[read_cnt].tpktTI_SATsec;
				}
			}

			if( bef_rec != -1 && aft_rec != -1 ) {
				break;
			}
			rec_cnt++;
		}

*/

	}

}



