/**********************************************************************

	ETI＝＞経過秒変換処理	                          種別：sub

    (1)処理概要
	    パラメータで指定したETIを衛星時刻に変換して通知する。

    (2)呼出形式
            int etc2es ( uint32_t eti_ext, uint64_t eti_base, double *elapsed_seconds )

    (3)パラメタ説明
	    ・ eti_ext (入力／uint32_t)
		   変換するETIを指定する。(単位：サイクル数)

	    ・ eti_base (入力／uint64_t)
		   Time Code の basic time とfractional time をそれぞれ
           上位4バイト(右詰めで不足上位桁は0詰め)と
           下位4バイト(左詰めで不足下位桁は0詰め)で指定

	    ・ elapsed_seconds (出力／double)
		   変換した衛星時刻が格納される。(単位：秒)

	    ・ 復帰値 (出力／int)
				RET_NORMAL					：正常終了
				RET_BEFORE_FIRST_RECORD		：校正結果が衛星時刻校正表の最初のレコード以前（最初のレコードからの外挿）
				RET_AFTER_LAST_RECORD		：校正結果が衛星時刻校正表の最終レコード以降（最終レコードからの外挿）
				RET_BETWEEN_DISCONT_RECORDS	：校正結果がTI不連続区間に該当（外挿した）
				RET_INAPPROPRIATE_PART		：指定etiの校正結果が別の連続区間にある
				RET_PARAM_ERR				：入力パラメータチェックエラー
				RET_FILE_UNSET_ERR			：ファイルパス未設定
				RET_FILE_OPEN_ERR			：ファイルオープンエラー
				RET_FILE_READ_ERR			：ファイルリードエラー
				RET_EXT_NOTFOUND_ERR		：指定ext無し
				RET_CALC_ERR				：算出不可

    (4) 作成日 2018/12/7  Ver3.3.0  jex)miyadera  フレーム方式対応





***********************************************************************/

#include	<stdio.h>
#include	<errno.h>
#include	<string.h>
#include	<stdlib.h>
#include	<inttypes.h>
#include    <math.h>
#include	"TimeCal_type.h"
#include	"SATCA_com.h"

//#define ETI_BASE64_MIN (2.32831E-10) /* pow(2.0, -32) */
//
//int etibase64equals( double a, double b ){
//    return (fabs(a-b) < ETI_BASE64_MIN) ? 1 : 0;
//}

int etc2es ( uint32_t eti_ext, uint64_t eti_base, double *elapsed_seconds )
{

	/****************************************************************/
	/* 領域定義 													*/
	/****************************************************************/
	static	TiTm_tbl	titm;	/* 基準時刻設定領域 */

	int		i, ir;
	double	ETI_TI;

	double	base_t;		/* ETI基本部の時刻計算用 拡張部(ext_t)と分けて計算するため */
	double	ext_t;		/* ETI拡張部の時刻計算用 */
	int		iret = RET_NORMAL;

	/* パラメータチェック */
	if( elapsed_seconds == NULL ) {
		return(RET_PARAM_ERR);
	}

	/***********************************************************/
	/*   基準ファイル設定処理　　　　　　　                    */
	/***********************************************************/
	ir = TiTm_Init_Eti2( &titm, eti_ext, eti_base );
	if( ir != RET_NORMAL){
#ifdef DBG_TITM
		printf("eti2es[%d] TiTm_Init_Eti Error!! return-code[%d] [%lu] [%llu]\n",__LINE__, ir, eti_ext, eti_base );
#endif
		return(ir);
	}
	
	/* 値変換 */
	double d1 = (double)( eti_base >> 32 );
	double d2 = (double)( eti_base & 0x00000000FFFFFFFFull) * pow(2.0, -32.0);
	double work_eti_base = d1 + d2;
	
#ifdef DBG_TITM	
	printf("etc2es[%d] etc2es : eti_base = 0x%016llx\n", __LINE__, eti_base);
	printf("etc2es[%d] etc2es : d1 = %lf\n", __LINE__, d1);
	printf("etc2es[%d] etc2es : d2 = %lf\n", __LINE__, d2);
	printf("etc2es[%d] etc2es : work_eti_base = %lf\n", __LINE__, work_eti_base);
#endif

	if(titm.SATCA_cnv_rdcnt == 0){
#ifdef DBG_TITM
		printf("eti2es[%d] SATCA_cnv_rdcnt = 0  [%d] \n", __LINE__, titm.SATCA_cnv_rdcnt );
#endif
		return(RET_CALC_ERR);
	}

	/****************************************************************/
	/* TIからUTへの変換処理											*/
	/****************************************************************/

	double rate = 0.0;
	int readrec = 0;




	/* 先頭レコードから 指定ETI拡張部が先頭レコードのよりも前 */
	if( eti_ext < titm.SATCA_cnv_tbl[0].ETI ){
		/* 有効レート探す */
		rate = find_valid_rate(0, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		if( work_eti_base > titm.SATCA_cnv_tbl[0].tpktTI ){
			base_t = ((titm.SATCA_cnv_tbl[0].tpktTI + titm.SATCA_Init.xffmax+1.0) - work_eti_base) * rate;
			ext_t = ((titm.SATCA_cnv_tbl[0].ETI - (eti_ext+1)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}else{
			base_t = (titm.SATCA_cnv_tbl[0].tpktTI - work_eti_base) * rate;
			ext_t = ((titm.SATCA_cnv_tbl[0].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}
		*elapsed_seconds = titm.SATCA_cnv_tbl[0].tpktTI_SATsec - ext_t - base_t;

		if( titm.SATCA_cnv_topflg == 1 ) {
			/* 校正表の先頭レコードより前 */
			return(RET_BEFORE_FIRST_RECORD);
		}
	/* 先頭レコードから 指定ETI拡張部が先頭レコードと同じ、ETI基本部が先頭レコードよりも前 */
	}else if( eti_ext == titm.SATCA_cnv_tbl[0].ETI && work_eti_base < titm.SATCA_cnv_tbl[0].tpktTI ){
		/* 有効レート探す */
		rate = find_valid_rate(0, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		base_t = (titm.SATCA_cnv_tbl[0].tpktTI - work_eti_base) * rate;
		*elapsed_seconds = titm.SATCA_cnv_tbl[0].tpktTI_SATsec - base_t;

		if( titm.SATCA_cnv_topflg == 1 ) {
			/* 校正表の先頭レコードより前 */
			return(RET_BEFORE_FIRST_RECORD);
		}
	/* 最終レコードから */
	}else if( eti_ext > titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI ){
		/* 有効レート探す */
		rate = find_valid_rate(titm.SATCA_cnv_rdcnt-1, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		if( work_eti_base < titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI ){
			base_t = ((work_eti_base + titm.SATCA_Init.xffmax+1.0) - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI) * rate;
			ext_t = ((eti_ext-1 - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI ) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}else{
			base_t = (work_eti_base - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI) * rate;
			ext_t = ((eti_ext - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI) * (titm.SATCA_Init.xffmax+1.0)) * rate;
		}
		*elapsed_seconds = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec + ext_t + base_t;

		if( titm.SATCA_cnv_endflg == 1 ) {
			/* 最終レコード+1よりも離れている場合はエラー */
			if( eti_ext > titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI+1 ){
				// error
				return(RET_EXT_NOTFOUND_ERR);
			}
			return(RET_AFTER_LAST_RECORD);
		}
	/* 最終レコードから */
	}else if( (eti_ext == titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI) && (titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI < work_eti_base)  ){
		/* 有効レート探す */
		rate = find_valid_rate(titm.SATCA_cnv_rdcnt-1, &titm);
		if( rate <= 0.0 ) {
			return(RET_CALC_ERR);
		}

		base_t = (work_eti_base - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI) * rate;
		*elapsed_seconds = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec + base_t;

		if( titm.SATCA_cnv_endflg == 1 ) {
			/* 校正表の先頭レコードより前 */
			return(RET_AFTER_LAST_RECORD);
		}
	}else{
		/* 読み込み済みの校正表レコード数分Loop */
		for( i=0; i<titm.SATCA_cnv_rdcnt-1; i++ ) {
			/* 過去レコード(i)とETI拡張部、ETI基本部が一致 */
			if( titm.SATCA_cnv_tbl[i].ETI == eti_ext && titm.SATCA_cnv_tbl[i].tpktTI == work_eti_base ){
//			if( titm.SATCA_cnv_tbl[i].ETI == eti_ext && etibase64equals(titm.SATCA_cnv_tbl[i].tpktTI, work_eti_base) ){
				*elapsed_seconds = titm.SATCA_cnv_tbl[i].tpktTI_SATsec;
				break;
			/* 未来レコード(i+1)とETI拡張部、ETI基本部が一致 */
			}else if( titm.SATCA_cnv_tbl[i+1].ETI == eti_ext && titm.SATCA_cnv_tbl[i+1].tpktTI == work_eti_base ){
//			}else if( titm.SATCA_cnv_tbl[i+1].ETI == eti_ext && etibase64equals(titm.SATCA_cnv_tbl[i+1].tpktTI, work_eti_base) ){
				*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec;
				break;
			}else if( titm.SATCA_cnv_tbl[i].ETI <= eti_ext && eti_ext <= titm.SATCA_cnv_tbl[i+1].ETI ){
				if( titm.SATCA_cnv_tbl[i].ETI == titm.SATCA_cnv_tbl[i+1].ETI ){
					if( titm.SATCA_cnv_tbl[i].tpktTI <= work_eti_base && work_eti_base <= titm.SATCA_cnv_tbl[i+1].tpktTI ){
						/* 未来レコード(i+1)から計算 */
						/* 有効レート探す */
						rate = find_valid_rate(i+1, &titm);
						if( rate <= 0.0 ) {
							return(RET_CALC_ERR);
						}

						/* ETI拡張部、ETI基本部を別々に計算し、差分の時間を求める */
						base_t = (titm.SATCA_cnv_tbl[i+1].tpktTI - work_eti_base) * rate;
						ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;

						*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - ext_t - base_t;
						break;
					}else{
						continue;
					}
				}else{	/* if(titm.SATCA_cnv_tbl[i].ETI < titm.SATCA_cnv_tbl[i+1].ETI ) */
					/* 連続区間でのロールオーバー判定 */
					if( titm.SATCA_cnv_tbl[i].ETI == eti_ext && strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ){
						/* 過去レコード(i)から計算 */
						/* 有効レート探す */
						rate = find_valid_rate(i, &titm);
						if( rate <= 0.0 ) {
							return(RET_CALC_ERR);
						}

						/* ETI拡張部、ETI基本部を別々に計算し、差分の時間を求める */
						base_t = (work_eti_base - titm.SATCA_cnv_tbl[i].tpktTI) * rate;
						ext_t = ((eti_ext - titm.SATCA_cnv_tbl[i].ETI) * (titm.SATCA_Init.xffmax+1.0)) * rate;

						*elapsed_seconds = titm.SATCA_cnv_tbl[i].tpktTI_SATsec + ext_t + base_t;

						/* 不連続区間の場合 */
						if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ) {
							/* 校正結果がTI-UTCペアの間にある */
							if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < *elapsed_seconds) && (*elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ){
								return(RET_BETWEEN_DISCONT_RECORDS);
							}else{
								return(RET_INAPPROPRIATE_PART);
							}
						}
						break;
					}else if( titm.SATCA_cnv_tbl[i+1].ETI == eti_ext ){
						if( work_eti_base < titm.SATCA_cnv_tbl[i+1].tpktTI ){
							/* 未来レコード(i+1)から計算 */
							/* 有効レート探す */
							rate = find_valid_rate(i+1, &titm);
							if( rate <= 0.0 ) {
								return(RET_CALC_ERR);
							}

							/* ETI拡張部、ETI基本部を別々に計算し、差分の時間を求める */
							base_t = (titm.SATCA_cnv_tbl[i+1].tpktTI - work_eti_base) * rate;
							ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;

							*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - ext_t - base_t;

							/* 不連続区間の場合 */
							if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ) {
								/* 校正結果がTI-UTCペアの間にある */
								if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < *elapsed_seconds) && (*elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ){
									return(RET_BETWEEN_DISCONT_RECORDS);
								}else{
									return(RET_INAPPROPRIATE_PART);
								}
							}
							break;
						}else{
							continue;
						}
					}else{
						/* 未来レコード(i+1)から計算 */
						/* 有効レート探す */
						rate = find_valid_rate(i+1, &titm);
						if( rate <= 0.0 ) {
							return(RET_CALC_ERR);
						}

						if( work_eti_base > titm.SATCA_cnv_tbl[i+1].tpktTI ){
							base_t = ((titm.SATCA_cnv_tbl[i+1].tpktTI + titm.SATCA_Init.xffmax+1.0) - work_eti_base) * rate;
							ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext+1)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
						}else{
							base_t = (titm.SATCA_cnv_tbl[i+1].tpktTI - work_eti_base) * rate;
							ext_t = ((titm.SATCA_cnv_tbl[i+1].ETI - (eti_ext)) * (titm.SATCA_Init.xffmax+1.0)) * rate;
						}
						*elapsed_seconds = titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - ext_t - base_t;

						/* 不連続区間の場合 */
						if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") == 0 ) {
							/* 校正結果がTI-UTCペアの間にある */
							if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < *elapsed_seconds) && (*elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ){
								return(RET_BETWEEN_DISCONT_RECORDS);
							}else{
								return(RET_INAPPROPRIATE_PART);
							}
						}
						break;
					}
				}
			}
		}
	}


	return( RET_NORMAL );


}


