/**********************************************************************

	経過秒⇒TI変換処理	                          種別：sub

    (1)処理概要
	    パラメータで指定した経過秒（UTC）をETI基本部、ETI拡張部に変換して通知する。

    (2)呼出形式
            int es2eti( const double elapsed_seconds, uint32_t *eti_base, uint32_t *eti_ext[, uint32_t extrapolation] )
            int es2ceti( const double elapsed_seconds, uint32_t *eti_base, uint32_t *eti_ext[, uint32_t extrapolation] )

    (3)パラメタ説明
	    ・ elapsed_seconds (入力／double)
		   変換する経過秒が格納される。(単位：秒)

	    ・ eti_base (出力／uint32_t)
		   変換した時刻 (TI) が格納される。値は31.25msec(2^-5sec)

	    ・ eti_ext (出力／uint32_t)
		   対応するETIが格納される。(単位：サイクル数)

		・ extrapolation（入力／uint32_t）
		   不連続時の処理基準とするEXTの方向。（1：過去側、2：未来側）
		   省略時：不連続区間であった場合、エラーとする。

	    ・ 復帰値 (出力／int)
				RET_NORMAL					：正常終了（可変引数(in_ext)を指定した場合は無視した）
				RET_BEFORE_FIRST_RECORD		：校正結果が衛星時刻校正表の最初のレコード以前（最初のレコードからの外挿）
				RET_AFTER_LAST_RECORD		：校正結果が衛星時刻校正表の最終レコード以降（最終レコードからの外挿）
				RET_BETWEEN_DISCONT_RECORDS	：校正結果がTI不連続区間に該当（外挿した）
				RET_BETWEEN_DISCONT_RECORDS_WITH_ROLLOVER
											：校正結果がTI不連続区間に該当し、ロールオーバーがあった（外挿した）
				RET_PARAM_ERR				：入力パラメータチェックエラー
				RET_FILE_UNSET_ERR			：ファイルパス未設定
				RET_FILE_OPEN_ERR			：ファイルオープンエラー
				RET_FILE_READ_ERR			：ファイルリードエラー
				RET_EXT_NOTFOUND_ERR		：指定ext無し
				RET_DISCONTINUITY_ERR		：不連続区間のため算出不可
											（可変引数(extrapolation)を指定しない場合で、不連続区間であった）
				RET_RUNOUT_OF_EXT_ERR		：
				RET_CALC_ERR				：算出不可

    (4) 作成日 2011/02/10
            2012/11/15  Ver2.0 y.yoneda  インデントの調整
            2012/11/16  Ver2.0 y.yoneda  静解析結果の修正
			2012/11/21  Ver2.0 y.yoneda  Ver2.0 対応
			2013/08/    Ver3.0.0 fips    ETI拡張部を含むETIの値を扱う際の桁落ちに対応

***********************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>			/* Add Ver2.0 */
#include	<errno.h>
#include	<string.h>
#include	<inttypes.h>
#include	"TimeCal_type.h"	/* Chg Ver2.0 */
#include	"SATCA_com.h"

int proc_es2eti( double elapsed_seconds, uint32_t *eti_base, uint32_t *eti_ext, uint32_t extrapolation);
int proc_es2ceti( double elapsed_seconds, double *eti_base, uint32_t *eti_ext, uint32_t extrapolation);

/* Add Ver2.0 Start */
int  func_es2eti( double elapsed_seconds,  uint32_t *eti_base, ... )
{
	int ret;

	/* 可変引数の取得 */
    uint32_t a;
	uint32_t *eti_ext;						/* 入力パラメータ（固定引数） */
    uint32_t extrapolation = (uint32_t)0;	/* 入力パラメータ（可変引数） */
	va_list arg;
    va_start(arg, eti_base);				/* eti_baseを最終仮引数として使用 */

	eti_ext = (uint32_t *)va_arg(arg, uint32_t *);
    a = va_arg(arg, uint32_t);
    if(a != UINT32_MAX){
        extrapolation = a;
    }
    va_end(arg);

	ret = proc_es2eti( elapsed_seconds, eti_base, eti_ext, extrapolation );

	return(ret);
}

int  func_es2ceti( double elapsed_seconds,  double *eti_base, ... )
{
	int ret;

	/* 可変引数の取得 */
    uint32_t a;
	uint32_t *eti_ext;						/* 入力パラメータ（固定引数） */
    uint32_t extrapolation = (uint32_t)0;	/* 入力パラメータ（可変引数） */
    va_list arg;
    va_start(arg, eti_base);				/* eti_baseを最終仮引数として使用 */

	eti_ext = (uint32_t *)va_arg(arg, uint32_t *);
    a = va_arg(arg, uint32_t);
    if(a != UINT32_MAX){
        extrapolation = a;
    }
    va_end(arg);

	ret = proc_es2ceti( elapsed_seconds, eti_base, eti_ext, extrapolation );

	return(ret);
}

int  proc_es2eti( double elapsed_seconds, uint32_t *eti_base, uint32_t *eti_ext, uint32_t extrapolation)
{
	double dti = 0.0;
	int ret;

	/* パラメータチェック */
	if( elapsed_seconds < 0 || eti_base == NULL || eti_ext == NULL ) {
		return(RET_PARAM_ERR);
	}

	/* 経過秒を指定し、TIのパケット単位系実数を取得する */
	ret = proc_es2ceti( elapsed_seconds, &dti, eti_ext, extrapolation );

	*eti_base = (uint32_t)dti;	/* パケットTI単位系整数に変換 */

	return(ret);
}
/* Add Ver2.0  End  */

/* Chg Ver2.0 */
int  proc_es2ceti( double elapsed_seconds, double *eti_base, uint32_t *eti_ext, uint32_t extrapolation)
{

	/****************************************************************/
	/* 領域定義 													*/
	/****************************************************************/
	static  TiTm_tbl        titm;   /* 基準時刻設定領域 */

	int	i, ir;
	double	ETI_EXT;	/* Chg Ver2.0 */
	double	ETI_BASE;	/* Add Ver2.0 */
	double	Tin = 0.0;	/* Chg Ver2.0 */
	uint32_t ext_wk;	/* Add Ver2.0 */


	/* ver3.0.0 add */
	uint32_t ext_diff;
	double base_diff;



	/* Add Ver2.0 パラメータチェック */
	if( elapsed_seconds < 0 || eti_base == NULL || eti_ext == NULL || extrapolation > 2 ) {
		return(RET_PARAM_ERR);
	}

	/***********************************************************/
	/*   衛星時刻校正表設定処理                                */
	/***********************************************************/
	ir = TiTm_Init_Sat( &titm, elapsed_seconds, (uint32_t)0, (uint32_t)0 );	/* Chg Ver3.0.0 */
	if( ir != 0){
#ifdef DBG_TITM
		printf("es2ceti[%d] TiTm_Init_Sat Error!! return-code[%d] [%lf]\n", __LINE__, ir, elapsed_seconds );
#endif
		return(ir);
	}

	/****************************************************************/
	/*  UTCからTIを求める											*/
	/****************************************************************/
	/* Add&Chg Ver2.0 Start 以降の処理は全体的に変更 */
	double rate = 0.0;
	int readrec = 0;
	double over_plus = 0;
	if( (titm.SATCA_cnv_tbl[0].ETI & ETI_FLAG_PATTERN) == 0 ) {
		over_plus = ((double)ETI_FLAG_MASK_PATTERN + 1.0 )* (titm.SATCA_Init.xffmax + 1.0);
	}
	else {
		over_plus = ((double)UINT32_MAX + 1.0) * (titm.SATCA_Init.xffmax + 1.0);
	}
	/*********************************************************************/
	/*【パターン1】指定された経過秒が時刻変換テーブルの範囲より過去      */
	/*********************************************************************/
	if( elapsed_seconds < titm.SATCA_cnv_tbl[0].tpktTI_SATsec ){

		/* ver3.0.0 change */
		/* 時間差からカウンタ値を計算 */
		base_diff = (titm.SATCA_cnv_tbl[0].tpktTI_SATsec - elapsed_seconds) / titm.SATCA_cnv_tbl[0].tpktTI_dT;
		/* ETI拡張部取得 */
		ext_diff = base_diff / (titm.SATCA_Init.xffmax + 1.0);
		/* ETI基本部取得 */
		base_diff = base_diff - ext_diff * (titm.SATCA_Init.xffmax + 1.0);


		/* ETI拡張部が0未満となる : エラー */
		if( (int)((titm.SATCA_cnv_tbl[0].ETI & ETI_FLAG_MASK_PATTERN) - ext_diff) < 0 ){
			return(RET_CALC_ERR);
		}else{
			/* 基本部が0未満 */
			if( titm.SATCA_cnv_tbl[0].tpktTI - base_diff < 0 ){
				/* 拡張部を-1すると0未満となる */
				if( ((titm.SATCA_cnv_tbl[0].ETI & ETI_FLAG_MASK_PATTERN) - ext_diff) == 0 ){
					return(RET_CALC_ERR);
				}else{
					*eti_ext = titm.SATCA_cnv_tbl[0].ETI - ext_diff - 1;
					*eti_base = (titm.SATCA_cnv_tbl[0].tpktTI + titm.SATCA_Init.xffmax + 1.0) - base_diff;
				}
			}else{
				*eti_ext = titm.SATCA_cnv_tbl[0].ETI - ext_diff;
				*eti_base = titm.SATCA_cnv_tbl[0].tpktTI - base_diff;
			}
		}

//		Tin = titm.SATCA_cnv_tbl[0].tpktETITI - (titm.SATCA_cnv_tbl[0].tpktTI_SATsec - elapsed_seconds)/titm.SATCA_cnv_tbl[0].tpktTI_dT;

#ifdef DBG_TITM
		printf("【パターン1】指定された経過秒が時刻変換テーブルの範囲より過去\n");
		printf(" es2ceti[%d] SATCA_cnv_tbl[0].tpktTI = %lf\n", __LINE__, titm.SATCA_cnv_tbl[0].tpktTI);
		printf(" es2ceti[%d] SATCA_cnv_tbl[0].tpktTI_dT  = %lf\n", __LINE__, titm.SATCA_cnv_tbl[0].tpktTI_dT);
		printf(" es2ceti[%d] ti[%lf][%x]\n", __LINE__, Tin, Tin);
#endif

//		if( Tin < 0.0 ){
#ifdef DBG_TITM
			printf(" 未来からのti計算結果 < 0\n");
#endif
//			return(RET_CALC_ERR);
//		}
//		ir = RET_BEFORE_FIRST_RECORD;

		return(RET_BEFORE_FIRST_RECORD);


	}
	/*********************************************************************/
	/*【パターン2】指定された経過秒が時刻変換テーブルの範囲より未来      */
	/*********************************************************************/
	else if( elapsed_seconds > titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec ){

		if( strcmp(titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].quality_flag,"E") == 0 ) {
			/* 最終レコードが不連続区間の場合、過去側の有効なレートを取得して計算 */
			readrec = titm.SATCA_cnv_rdcnt-2;
			while( readrec > 0 ) {
				if( strcmp(titm.SATCA_cnv_tbl[readrec].quality_flag,"E") != 0 ) {
					rate = titm.SATCA_cnv_tbl[readrec].tpktTI_dT;
					break;
				}
				readrec--;
			}
			if( rate <= 0.0 ) {
				return(RET_CALC_ERR);
			}

			/* ver3.0.0 change */
			/* 時間差からカウンタ値を計算 */
			base_diff = (elapsed_seconds - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec) / rate;
			/* ETI拡張部取得 */
			ext_diff = base_diff / (titm.SATCA_Init.xffmax + 1.0);
			/* ETI基本部取得 */
			base_diff = base_diff - ext_diff * (titm.SATCA_Init.xffmax + 1.0);


//			Tin = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktETITI - (titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec - elapsed_seconds) / rate;
		}
		else {
			/* ver3.0.0 change */
			/* 時間差からカウンタ値を計算 */
			base_diff = (elapsed_seconds - titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec) / titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_dT;
			/* ETI拡張部取得 */
			ext_diff = base_diff / (titm.SATCA_Init.xffmax + 1.0);
			/* ETI基本部取得 */
			base_diff = base_diff - ext_diff * (titm.SATCA_Init.xffmax + 1.0);

//			Tin = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktETITI - (titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec - elapsed_seconds)/titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_dT;
		}


		/* ETI拡張部がフラグ以外の領域を振り切る : エラー */
		if( ((titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI & ETI_FLAG_MASK_PATTERN) + ext_diff) > ETI_FLAG_MASK_PATTERN ){
			return(RET_CALC_ERR);
		}else{
			/* 基本部が最大超える */
			if( titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI + base_diff > titm.SATCA_Init.xffmax ){
				/* 拡張部を+1するとフラグ以外の領域を振り切る */
				if( ((titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI & ETI_FLAG_MASK_PATTERN) + ext_diff) == ETI_FLAG_MASK_PATTERN ){
					return(RET_CALC_ERR);
				}else{
					*eti_ext = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI + ext_diff + 1;
					*eti_base = (titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI + base_diff) - (titm.SATCA_Init.xffmax + 1.0);
				}
			}else{
				*eti_ext = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].ETI + ext_diff;
				*eti_base = titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI + base_diff;
			}
		}

//		if( Tin < 0 ) {
//			Tin += over_plus;
//		}
#ifdef DBG_TITM
		printf("【パターン2】指定された経過秒が時刻変換テーブルの範囲より未来\n");
		printf(" es2ceti[%d] titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec = %lf\n", __LINE__, titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_SATsec);
		printf(" es2ceti[%d]   titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI = %lf\n", __LINE__, titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI);
		printf(" es2ceti[%d]   titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_dT  = %lf\n", __LINE__, titm.SATCA_cnv_tbl[titm.SATCA_cnv_rdcnt-1].tpktTI_dT);
		printf(" es2ceti[%d] ti[%lf][%x]\n", __LINE__, Tin, Tin);
#endif
//		ir = RET_AFTER_LAST_RECORD;		/* 最終レコードより後ろ */

		return(RET_AFTER_LAST_RECORD);

	}
	/*********************************************************************/
	/*【パターン3】指定された経過秒が時刻変換テーブルの範囲内            */
	/*********************************************************************/
	else {
#ifdef DBG_TITM
		printf("【パターン3】指定された経過秒が時刻変換テーブルの範囲内\n");
#endif
		/* レコード数分ループ */
		for( i=0; i<=titm.SATCA_cnv_rdcnt-1; i++ ){
#ifdef DBG_TITM
			printf(" es2ceti[%d] SATCA_cnv_tbl[%d].tpktTI_SATsec = %lf\n", __LINE__, i, titm.SATCA_cnv_tbl[i].tpktTI_SATsec);
#endif
			if( elapsed_seconds == titm.SATCA_cnv_tbl[i].tpktTI_SATsec ){
#ifdef DBG_TITM
				printf(" テーブル内の%dレコード目に時刻があった！！\n", i+1);
				printf(" Time2Ti[%d]              elapsed_seconds = %lf\n", __LINE__, elapsed_seconds);
				printf(" Time2Ti[%d] SATCA_cnv_tbl[%d].tpktTI_SATsec = %lf\n", __LINE__, i, titm.SATCA_cnv_tbl[i].tpktTI_SATsec);
				printf(" Time2Ti[%d]   SATCA_cnv_tbl[%d].tpktTI = %lf\n", __LINE__, i, titm.SATCA_cnv_tbl[i].tpktTI);
#endif

				/* ver3.0.0 change */
				*eti_ext = titm.SATCA_cnv_tbl[i].ETI;
				*eti_base = titm.SATCA_cnv_tbl[i].tpktTI;

//				Tin = titm.SATCA_cnv_tbl[i].tpktETITI;
				ir = RET_NORMAL;
				break;
			}

			if( (titm.SATCA_cnv_tbl[i].tpktTI_SATsec < elapsed_seconds) &&
									 (elapsed_seconds < titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec) ) {
#ifdef DBG_TITM
				printf("【パターン3】指定された経過秒が時刻変換テーブルの範囲内\n");
				printf(" es2ceti[%d] SATCA_cnv_tbl[i+1].tpktTI = %lf\n", __LINE__, titm.SATCA_cnv_tbl[i+1].tpktTI);
				printf(" es2ceti[%d] SATCA_cnv_tbl[i+1].tpktTI_dT  = %lf\n", __LINE__, titm.SATCA_cnv_tbl[i+1].tpktTI_dT);
#endif
				/* レコードクォリティフラグが"E"ではない場合 */
				if( strcmp(titm.SATCA_cnv_tbl[i+1].quality_flag,"E") != 0 ) {

					/* ver3.0.0 change */
					/* 時間差からカウンタ値を計算 */
					base_diff = (titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - elapsed_seconds) / titm.SATCA_cnv_tbl[i+1].tpktTI_dT;
					/* ETI拡張部取得 */
					ext_diff = base_diff / (titm.SATCA_Init.xffmax + 1.0);
					/* ETI基本部取得 */
					base_diff = base_diff - ext_diff * (titm.SATCA_Init.xffmax + 1.0);


					/* 基本部が0未満 */
					if( titm.SATCA_cnv_tbl[i+1].tpktTI - base_diff < 0 ){
						*eti_ext = titm.SATCA_cnv_tbl[i+1].ETI - ext_diff - 1;
						*eti_base = (titm.SATCA_cnv_tbl[i+1].tpktTI + titm.SATCA_Init.xffmax + 1.0) - base_diff;
					}else{
						*eti_ext = titm.SATCA_cnv_tbl[i+1].ETI - ext_diff;
						*eti_base = titm.SATCA_cnv_tbl[i+1].tpktTI - base_diff;
					}


//					Tin = titm.SATCA_cnv_tbl[i+1].tpktETITI - (titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - elapsed_seconds)/titm.SATCA_cnv_tbl[i+1].tpktTI_dT;
//					if( Tin < 0 ) {
//						Tin += over_plus;
//					}
					ir = RET_NORMAL;

				}
				else {
#ifdef DBG_TITM
					printf(" レコードクオリティフラグ[%s]\n", titm.SATCA_cnv_tbl[i+1].quality_flag);
#endif
					if( extrapolation == 1 ) {
						/* 過去側のTI-UTCペアから外挿 */
						if( strcmp(titm.SATCA_cnv_tbl[i].quality_flag,"E") == 0 ) {
							/* 過去側から不連続が連続していた場合、過去側の有効なレートを取得して計算 */
							rate = 0.0;
							readrec = i-1;
							while( readrec > 0 ) {
								if( strcmp(titm.SATCA_cnv_tbl[readrec].quality_flag,"E") != 0 ) {
									rate = titm.SATCA_cnv_tbl[readrec].tpktTI_dT;
									break;
								}
								readrec--;
							}
							if( rate <= 0.0 ) {
								return(RET_CALC_ERR);
							}

//							Tin = titm.SATCA_cnv_tbl[i].tpktETITI - (titm.SATCA_cnv_tbl[i].tpktTI_SATsec - elapsed_seconds) / rate;
						}
						else {

							rate = titm.SATCA_cnv_tbl[i].tpktTI_dT;

//							Tin = titm.SATCA_cnv_tbl[i].tpktETITI - (titm.SATCA_cnv_tbl[i].tpktTI_SATsec - elapsed_seconds)/titm.SATCA_cnv_tbl[i].tpktTI_dT;
						}

						/* ver3.0.0 change */
						/* 時間差からカウンタ値を計算 */
						base_diff = (elapsed_seconds - titm.SATCA_cnv_tbl[i].tpktTI_SATsec) / rate;
						/* ETI拡張部取得 */
						ext_diff = base_diff / (titm.SATCA_Init.xffmax + 1.0);
						/* ETI基本部取得 */
						base_diff = base_diff - ext_diff * (titm.SATCA_Init.xffmax + 1.0);

						/* 基本部が最大超える */
						if( titm.SATCA_cnv_tbl[i].tpktTI + base_diff > titm.SATCA_Init.xffmax ){
							*eti_ext = titm.SATCA_cnv_tbl[i].ETI + ext_diff + 1;
							*eti_base = (titm.SATCA_cnv_tbl[i].tpktTI + base_diff) - (titm.SATCA_Init.xffmax + 1.0);
							ir = RET_BETWEEN_DISCONT_RECORDS_WITH_ROLLOVER;
						}else{
							*eti_ext = titm.SATCA_cnv_tbl[i].ETI + ext_diff;
							*eti_base = titm.SATCA_cnv_tbl[i].tpktTI + base_diff;
							ir = RET_BETWEEN_DISCONT_RECORDS;
						}

//						if( Tin < 0 ) {
//							Tin += over_plus;
//						}
//						ETI_EXT = Tin / (titm.SATCA_Init.xffmax + 1.0);
//						ext_wk = (uint32_t)ETI_EXT;
//						ETI_BASE  = Tin - ext_wk * (titm.SATCA_Init.xffmax + 1.0);
//						if( titm.SATCA_cnv_tbl[i].tpktTI > ETI_BASE ) {
//							ir = RET_BETWEEN_DISCONT_RECORDS_WITH_ROLLOVER;
//						}
//						else {
//							ir = RET_BETWEEN_DISCONT_RECORDS;
//						}

					}
					else if ( extrapolation == 2 ) {
						/* 未来側のTI-UTCペアから外挿 */
						if( i+1 == titm.SATCA_cnv_rdcnt-1 ) {
							/* 最終レコードが不連続区間の場合、外挿計算できないので算出不可とする */
							return(RET_CALC_ERR);
						}
						else if( (strcmp(titm.SATCA_cnv_tbl[i+2].quality_flag,"E") == 0) ) {
							/* 未来側に不連続が連続していた場合、未来側の有効なレートを取得して計算 */
							rate = 0.0;
							readrec = i+3;
							while( readrec < titm.SATCA_cnv_rdcnt-1 ) {
								if( strcmp(titm.SATCA_cnv_tbl[readrec].quality_flag,"E") != 0 ) {
									rate = titm.SATCA_cnv_tbl[readrec].tpktTI_dT;
									break;
								}
								readrec++;
							}
							if( rate <= 0.0 ) {
								return(RET_CALC_ERR);
							}
//							Tin = titm.SATCA_cnv_tbl[i+1].tpktETITI - (titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - elapsed_seconds) / rate;
						}
						else {

							rate = titm.SATCA_cnv_tbl[i+2].tpktTI_dT;

//							Tin = titm.SATCA_cnv_tbl[i+1].tpktETITI - (titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - elapsed_seconds)/titm.SATCA_cnv_tbl[i+2].tpktTI_dT;
						}

						/* ver3.0.0 change */
						/* 時間差からカウンタ値を計算 */
						base_diff = (titm.SATCA_cnv_tbl[i+1].tpktTI_SATsec - elapsed_seconds) / rate;
						/* ETI拡張部取得 */
						ext_diff = base_diff / (titm.SATCA_Init.xffmax + 1.0);
						/* ETI基本部取得 */
						base_diff = base_diff - ext_diff * (titm.SATCA_Init.xffmax + 1.0);

						/* 基本部が0未満 */
						if( titm.SATCA_cnv_tbl[i+1].tpktTI - base_diff < 0 ){
							*eti_ext = titm.SATCA_cnv_tbl[i+1].ETI - ext_diff - 1;
							*eti_base = (titm.SATCA_cnv_tbl[i+1].tpktTI + titm.SATCA_Init.xffmax + 1.0) - base_diff;
							ir = RET_BETWEEN_DISCONT_RECORDS_WITH_ROLLOVER;
						}else{
							*eti_ext = titm.SATCA_cnv_tbl[i+1].ETI - ext_diff;
							*eti_base = titm.SATCA_cnv_tbl[i+1].tpktTI - base_diff;
							ir = RET_BETWEEN_DISCONT_RECORDS;
						}

//						if( Tin < 0 ) {
//							Tin += over_plus;
//						}
//						ETI_EXT = Tin / (titm.SATCA_Init.xffmax + 1.0);
//						ext_wk = (uint32_t)ETI_EXT;
//						ETI_BASE  = Tin - ext_wk * (titm.SATCA_Init.xffmax + 1.0);
//						if( titm.SATCA_cnv_tbl[i+1].tpktTI < ETI_BASE ) {
//							ir = RET_BETWEEN_DISCONT_RECORDS_WITH_ROLLOVER;
//						}
//						else {
//							ir = RET_BETWEEN_DISCONT_RECORDS;
//						}

					}
					else {						/* 不連続区間における処理基準extrapolation指定なし */
						return(RET_DISCONTINUITY_ERR);
					}
				}
				break;
			}
		}
	}

	/* TimePacketTIのベース値と2ndHeaderのベース値を可変とする */

//	ETI_EXT = Tin / (titm.SATCA_Init.xffmax + 1.0);
//	ext_wk = (uint32_t)ETI_EXT;
	if( (titm.SATCA_cnv_tbl[0].ETI & ETI_FLAG_PATTERN) == 0 ) {
		/* ETI flag off */
#ifdef RET_RUNOUT_OF_EXT_ERR
		/* if( (ext_wk & ETI_FLAG_PATTERN) != 0 ) { */
		if( (*eti_ext & ETI_FLAG_PATTERN) != 0 ) {
			return(RET_RUNOUT_OF_EXT_ERR);
		}
#endif
		*eti_ext = *eti_ext & ETI_FLAG_MASK_PATTERN;
	}
	else {
		/* ETI flag on */
#ifdef RET_RUNOUT_OF_EXT_ERR
		if( (*eti_ext & ETI_FLAG_PATTERN) == 0 ) {
			return(RET_RUNOUT_OF_EXT_ERR);
		}
#endif
		*eti_ext = *eti_ext | ETI_FLAG_PATTERN;
	}
//	*eti_base  = Tin - ext_wk * (titm.SATCA_Init.xffmax + 1.0);
//#ifdef DBG_TITM
//	printf(" 内部関数のETi計算結果(double)  [%lf]\n", Tin );
//	printf(" 内部関数のTi計算結果(double)[%lf]\n",  *eti_base );
//	printf(" 内部関数のETi計算結果(uint32_t)[%u]\n",  *eti_ext );
//#endif

	return(ir);
	/* Add&Chg Ver2.0  End  */

}
