/**********************************************************************

	Ti⇒経過秒変換処理	                          種別：sub

    (1)処理概要
	    パラメータで指定したTIを衛星時刻に変換して通知する。
	    ti2es関数の後方互換関数。

    (2)呼出形式
            int Ti2Time ( const char *Date_Time, uint32_t Ti,
                                    double *Sat_Time, uint16_t *Ext )

    (3)パラメタ説明
	    ・ Date_Time (入力／char)
		   処理するデータの時刻を指定する。(YYYYMMDDhhmmss)

	    ・ Ti (入力／uint32_t)
		   変換する時刻 (TI) を指定する。値は31.25msec(2^-5sec)

	    ・ Sat_Time (出力／double)
		   変換した衛星時刻が格納される。(単位：秒)

	    ・ Ext (出力／uint16_t)
		   対応するETIが格納される。(単位：サイクル数)

	    ・ 復帰値 (出力／int)
				0			：正常終了
				-1			：入力パラメータチェックエラー
				-11			：ファイルオープンエラー
				-12			：ファイルリードエラー
				-99			：算出不可

    (4) 作成日 2013/06/01  後方互換関数として新規作成

***********************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<errno.h>
#include	<string.h>
#include	<inttypes.h>
#include	"SATCA_com.h"
#include	"TiTm_type.h"

int timeCalFileSetFlagCheck();

int Ti2Time(const char *Date_Time, uint32_t Ti, double *Sat_Time, uint16_t *Ext)
{

	/****************************************************************/
	/* 領域定義														*/
	/****************************************************************/
	char *timecal_file_path, *leap_file_path;	/* ファイルパス */
	int ret, f_ret;
	uint32_t const extMask = 0x7fff8000;	/* ti2esから返却されたETI拡張部上位16ビットの確認用 */
	uint32_t const extFlgMask = 0x80000000;
	uint16_t const extFlgSet = 0x8000;
	uint32_t extVal;

	/*
	 * 衛星時刻校正表ファイルパス確認
	 */
	if((timecal_file_path = getenv(ENV_TIMECAL_FILE_PATH)) == NULL ){
		/* 環境変数取得エラー */
		return(-11);
	}

	/*
         * うるう秒のファイルパス確認
         */
        if((leap_file_path = getenv(ENV_LEAP_FILE_PATH)) == NULL ){
                /* 環境変数取得エラー */
                return(-11);
        }

	/****************************************************************
	 * 衛星時刻校正表ファイルパスを設定済みか確認
	 ****************************************************************/
	ret = timeCalFileSetFlagCheck(0);
	if( ret == 0 ){
		/* ファイル設定 */
		ret = setTimeCalFile(timecal_file_path);
		/* 戻り値設定 */
		if( ret != RET_NORMAL ){
			return( -11 );
		}
		ret = timeCalFileSetFlagCheck(1);
	}

	/****************************************************************
         * うるう秒ファイルパスを設定済みか確認 *
         ****************************************************************/
        ret = leapSecFileSetFlagCheck(0);
        if( ret == 0 ){
                ret = setLeapSecFile(leap_file_path);
                /* 戻り値設定 */
                if( ret != RET_NORMAL ){
                        return( -11 );
                }
		ret = leapSecFileSetFlagCheck(1);
        }

	/****************************************************************
	 * ti2es関数呼び出し（ETI拡張部の領域はローカル変数を設定し、後で返却値に設定
	 ****************************************************************/
	ret = ti2es(Date_Time, Ti, Sat_Time, &extVal, TI2TIME_SRCH_TERM);

	/* 戻り値判定 */
	if( ret == RET_NORMAL ||
		ret == RET_BEFORE_FIRST_RECORD ||
		ret == RET_AFTER_LAST_RECORD ||
		ret == RET_BETWEEN_DISCONT_RECORDS ){
		/* ETI拡張部の上位16ビットが0でない場合はエラー */
		if( (extVal & extMask) != 0 ){
			f_ret = -99;
		}else{
			/* ETI拡張部に問題なければ、返却値に設定 */
			if( (extVal & extFlgMask) == 0 ){
				/* ETIフラグが0の場合 */
				*Ext = (uint16_t)extVal;
			}else{
				/* ETIフラグが1の場合 返却する値にフラグ設定 */
				*Ext = (uint16_t)(extVal | extFlgSet);
			}
			f_ret = 0;
		}
	}else if( ret == RET_PARAM_ERR ){
		f_ret = -1;
	}else if( ret == RET_FILE_UNSET_ERR || ret == RET_FILE_OPEN_ERR ){
		f_ret = -11;
	}else if( ret == RET_FILE_READ_ERR ){
		f_ret = -12;
	}else{
		f_ret = -99;
	}

	return( f_ret );

}

/*
 * 衛星時刻校正表のファイルパス設定済み／未設定判定
 * 
 * param:
 *    chkFlag     0:チェックのみ
 *                1:設定
 */
int timeCalFileSetFlagCheck( int chkFlag ){

	static int fileSetFlag = 0;

	if( chkFlag == 1){
		fileSetFlag = 1;
	}
	return(fileSetFlag);
}

