/******************************************************************

時刻付関数(初期処理)	                       種別：sub

    (1)処理概要
		予めファイル転送された時刻変換テーブルより、パラメータ
		で指定された衛星時刻付近のデータをプログラム内の領域
		に展開する。

    (2)呼出形式
	    Ir = TiTm_Init_Sat( *titm, In_SATsec, day )
	    Ir = TiTm_Init_Eti( *titm, ETI, TI )
	    Ir = TiTm_Init_Eti2(*titm, ETI, TI )

    (3)環境変数説明	----- Ver2.0 以降未使用
	    ・ TM_STD_FPATH
		    変換テーブルのあるファイル名をフルパスで指定する。

    (4)パラメタ説明
	    ・ In_SATsec (入力／double)
		   処理するデータの衛星時刻

	    ・ day (入力／uint32_t)
		   時刻校正表の検索範囲

	    ・ ETI (入力／uint32_t)
		   処理するデータのETI

	    ・ TI (入力／uint32_t)
		   処理するデータのTI

	    ・ TI (入力／uint64_t)
		   処理するデータのTI

	    ・復帰値(出力／int)
			RET_NORMAL					：正常終了
			RET_PARAM_ERR				：入力パラメータチェックエラー
			RET_FILE_UNSET_ERR			：ファイルパス未設定
			RET_FILE_OPEN_ERR			：ファイルオープンエラー
			RET_FILE_READ_ERR			：ファイルリードエラー

    (5) 作成日 2004/07/26
        修正日 2011/02/14 ファイルを共通フォーマットとする
               2012/02/17 時刻基準ファイルの２行目にコメント行追加
               2012/11/15 Ver2.0 y.yoneda  インデントの調整
               2012/11/16 Ver2.0 y.yoneda  静解析結果の修正
			   2012/11/21 Ver2.0 y.yoneda  Ver2.0 対応
			   2013/08/   Ver3.0.0 fips    レコード長チェック、エラーメッセージ出力追加
			   2014/09/01 Ver3.0.2 fae     変数初期化
			   2015/03/03 Ver3.1.0 fae     outputMessageチェック追加
			   2015/10/19 Ver3.1.1 fae     時刻基準ファイル読み込み処理修正
			   2018/12/07 Ver3.3.0 jex     フレーム方式対応

    (6) 作成者 山口 喬

******************************************************************/
#include	<stdio.h>
#include	<stdlib.h>
#include	<sys/fcntl.h>
#include	<time.h>
#include	<math.h>
#include	<errno.h>
#include	<string.h>
#include	<sys/stat.h>		/* Add Ver3.1.1 */
#include	"TimeCal_type.h"	/* Chg Ver2.0 */
#include	"SATCA_com.h"

/*****   内部関数プロトタイプ宣言   *****/
int TiTm_Init_sub( int inknd, TiTm_tbl *titm, double In_arg1, double In_arg2, double In_arg3 );	/* Chg Ver3.0.0 */
int TiTmInt_headset( char *read_rd, TiTm_tbl *titm );							/* Chg Ver2.0 */
void TiTmInt_recdiv( char *read_rd, SATCA_cnv *rec_tbl,  TiTm_tbl *titm );		/* Chg Ver2.0 */
/* Chg Ver3.1.1 start */
#if 0
long TiTmInt_fseek( int inknd, FILE *fp, double objval, long *last_lpos, TiTm_tbl *titm );
#else
int TiTmInt_fseek( int inknd, SATCA_cnv *SATCA_cnv_tbl, double objval, int tbl_cnt, TiTm_tbl *titm );
#endif
/* Chg Ver3.1.1 end   */
int TiTmInt_dataset( int tmcal_fsize, SATCA_cnv **SATCA_cnv_tbl, TiTm_tbl *titm );	/* Add Ver3.1.1 */
/****************************************/


int TiTm_Init_Sat( TiTm_tbl *titm, double In_SATsec, uint32_t day, uint32_t day2 )	/* Chg Ver3.0.0 */
{

	return(  TiTm_Init_sub( 1, titm, In_SATsec, (double)day, (double)day2 ) );	/* Chg Ver3.0.0 */
}


int TiTm_Init_Eti( TiTm_tbl *titm, uint32_t ETI, uint32_t TI )		/* Chg Ver2.0 */
{

	double In_ETI;		/* ETI */
	double In_TI;		/* TI  */

	In_ETI = (double)ETI;
	In_TI  = (double)TI;

	return(  TiTm_Init_sub( 2, titm, In_ETI, In_TI, 0.0 ) );		/* Chg Ver3.0.0 */
}

/* Add Ver3.3.0 start */
int TiTm_Init_Eti2( TiTm_tbl *titm, uint32_t ETI, uint64_t TI )
{

	double In_ETI;		/* ETI */
	double In_TI;		/* TI  */
	
	double d1 = (double)( TI >> 32 );
	double d2 = (double)( TI & 0x00000000FFFFFFFF) * pow(2.0, -32.0);

	In_ETI = (double)ETI;
	In_TI  = d1 + d2;

	return(  TiTm_Init_sub( 2, titm, In_ETI, In_TI, 0.0 ) );
}
/* Add Ver3.3.0 end */

/******************************************************************

	メインの処理を行う

	TiTm_Init_sub( int inknd, TiTm_tbl *titm, double In_arg1, double In_arg2 )

		in  int 		inknd  処理種別
		out TiTm_tbl 	*titm  抽出領域
		in  double 		In_arg1  基準値１(inknd=1:衛星時刻        2:ETI)
		in  double 		In_arg2  基準値２(inknd=1:検索範囲（過去(未来)n日） 2:TI)
		in  double 		In_arg3  基準値３(inknd=1:検索範囲（未来n日） 2:N/A)

	Chg Ver3.0.0 In_atg3追加
	Chg Ver3.0.1 変数初期化

******************************************************************/
int TiTm_Init_sub( int inknd, TiTm_tbl *titm, double In_arg1, double In_arg2, double In_arg3 )
{
	/****************************************************************/
	/* 領域定義														*/
	/****************************************************************/

//	FILE *fp;			/* ファイルポインタ 		*//* Del Ver3.1.1 */

//	char read_rd[TBASE_REC_LENG+1];		/* 変換ファイル１レコード分を格納 Chg ver3.0.0 *//* Del Ver3.1.1 */
//	char read_head[TBASE_REC_LENG+1];	/* ヘッダを格納 Chg ver3.0.0 *//* Del Ver3.1.1 */
	int i, rtn;
	int	rec_pos, top_pos, last_pos;
//	long	lpos, last_lpos;	/* Del Ver3.1.1 */

	time_t	clt_time;
	char fname[FILE_PATH_MAX_LENGTH+1];	/* Chg Ver2.0 */

//	SATCA_cnv rec_tbl;		/* 項目変換一時領域 *//* Del Ver3.1.1 */

	double In_ETI_TI;		/* ETI+TI */
//	double Etibefore;		/* 衛星時刻の7日前 		*//* Del Ver2.0 */
//	double Etiafter;		/* 衛星時刻の2日後 		*//* Del Ver2.0 */
	double SATstart, SATend;		/* 検索範囲の衛星時刻 		*//* Add Ver2.0 */

	uint32_t day = (uint32_t)0;		/* Add Ver2.0 */
	uint32_t day2 = (uint32_t)0;	/* Add Ver3.0.0 */
//	char errMsg[EMGS_MSG_LENGTH];	/* Add Ver3.0.0 エラーメッセージ *//* Del Ver3.1.1 */

	static off_t tmcal_fsize = 0;			/* Add Ver3.1.1 */
	static int cnv_tbl_cnt = 0;				/* Add Ver3.1.1 */
	static SATCA_cnv *SATCA_cnv_tbl = NULL;	/* Add Ver3.1.1 */

	/****************************************************************/
	/***                変換テーブルの読込み                      ***/
	/****************************************************************/

	/* Chg Ver3.0.0 以降、全体的にレコード長変更 TBASE_REC_LENG -> titm->SATCA_cnv_rdlen */

	/* Add&Chg Ver2.0 start */
	if( titm == NULL ) {
		return(RET_PARAM_ERR);
	}
#ifdef DBG_TITM
	printf( "[%s:%d] TiTm_Init START[%d][%lf][%lf]\n", __FILE__ , __LINE__, inknd, In_arg1, In_arg2 );
#endif

	/* Add&Chg Ver2.0 start */
	if( inknd == 1 ) {					/* Chg Ver3.0.0 In_arg2 == 0削除 */
		day = (uint32_t)In_arg2;
		day2 = (uint32_t)In_arg3;		/* Add Ver3.0.0 */
	}

	time( &clt_time ); /* Add Ver3.1.1 */

	memset(fname, 0x00, sizeof(fname));	/* Add Ver3.0.2 */
	rtn = filepath_mng(TMCAL_FILE_PATH_GET, fname);
	if( rtn == RET_FILE_UNSET_ERR ) {
#ifdef DBG_TITM
		printf( "file path unset error :TMCAL_FILE_PATH_GET\n" );
#endif
		return(RET_FILE_UNSET_ERR);
	}
	else if( rtn == RET_FILE_INIT_READ || titm->SATCA_initial != 123454321 ) {
	/* Add&Chg Ver2.0 end *//* 衛星時刻校正表初期情報クリア */
		memset(titm, 0x00, sizeof(TiTm_tbl));
		strcpy(titm->SATCA_Init.fname, fname);
	} else {
		if( inknd == 1 ) {
			/* Add Ver2.0 Start */
			SATstart = In_arg1;
			SATend = In_arg1;
			if( ((int)day != 0) || ((int)day2 != 0) ) {	/* Chg Ver3.0.0 add || ((int)day2 != 0) */
				SATstart -= (double)day * DAY_SEC;
				SATend += (double)day2 * DAY_SEC;	/* Chg Ver3.0.0 day -> day2*/;
			}
			/* Add Ver2.0  End  */
			if( (titm->SATCA_cnv_tbl[0].tpktTI_SATsec <= SATstart ) && (SATend <= titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt-1].tpktTI_SATsec ) ){	/* Chg Ver2.0 */
#ifdef DBG_TITM
				printf( "[%s:%d] TiTm_Init return1[%d][%lf]\n", __FILE__ , __LINE__, inknd, In_arg1 );
#endif
				return(RET_NORMAL);
			} else if( (titm->SATCA_cnv_topflg == 1) && (titm->SATCA_cnv_tbl[0].tpktTI_SATsec > SATend) ) {	/* Chg Ver2.0 */
				return(RET_NORMAL);
			} else if( (titm->SATCA_cnv_endflg == 1) && (titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt-1].tpktTI_SATsec < SATstart ) ) {	/* Chg Ver2.0 */
			/*	time( &clt_time );	Del Ver3.1.1 */
				if( clt_time < (titm->SATCA_cnv_getTime+TMCAL_FILE_CHK_SEC) ) {	/* Chg Ver3.1.1 2*60 -> TMCAL_FILE_CHK_SEC */
#ifdef DBG_TITM
					printf("[%s:%d]TIME not OVER\n", __FILE__, __LINE__ );
#endif
					return(RET_NORMAL);
				}
#ifdef DBG_TITM
				printf("[%s:%d]TIME OVER\n", __FILE__, __LINE__ );
#endif
			}
		} else {
			In_ETI_TI = In_arg1 * (titm->SATCA_Init.xffmax+1.0) + In_arg2;	/* Chg Ver2.0 */
			if( (titm->SATCA_cnv_tbl[0].tpktETITI <= In_ETI_TI) && (In_ETI_TI <= titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt-1].tpktETITI) ) {
#ifdef DBG_TITM
				printf( "[%s:%d] TiTm_Init return2[%d][%lf][%lf][%lf]    \n", __FILE__ , __LINE__, inknd, In_arg1, In_arg2, In_ETI_TI );
#endif
				return(RET_NORMAL);
			} else if( (titm->SATCA_cnv_topflg == 1) && (titm->SATCA_cnv_tbl[0].tpktETITI > In_ETI_TI ) ) {
				return(RET_NORMAL);
			} else if( (titm->SATCA_cnv_endflg == 1) && (titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt-1].tpktETITI < In_ETI_TI ) ) {
			/*	time( &clt_time );	Del Ver3.1.1 */
				if( clt_time < (titm->SATCA_cnv_getTime+TMCAL_FILE_CHK_SEC) ) {	/* Chg Ver3.1.1 2*60 -> TMCAL_FILE_CHK_SEC */
#ifdef DBG_TITM
					printf("[%s:%d]TIME not OVER\n", __FILE__, __LINE__ );
#endif
					return(RET_NORMAL);
				}
#ifdef DBG_TITM
				printf("[%s:%d]TIME OVER\n", __FILE__, __LINE__ );
#endif
			}
		}
	}

	/* Add Ver3.1.1 start */	
	/* 衛星時刻校正表の再読込判定 */
	if( clt_time >= (titm->SATCA_cnv_getTime+TMCAL_FILE_CHK_SEC) ) {
		struct stat buf;
		if(stat(titm->SATCA_Init.fname, &buf) != 0){
#ifdef DBG_TITM
			printf("[%s:%d] TIMEBASE_FILE Stat Error!! [%s] \n", __FILE__, __LINE__, fname );
#endif
			return(RET_FILE_OPEN_ERR);
		}
		if(tmcal_fsize != buf.st_size){
#ifdef DBG_TITM
			printf("[%s:%d]Reading of a TIMEBASE_FILE is needed.\n", __FILE__, __LINE__ );
#endif
			/* 衛星時刻校正表初期情報クリア */
			memset(titm, 0x00, sizeof(TiTm_tbl));
			strcpy(titm->SATCA_Init.fname, fname);
			if(SATCA_cnv_tbl != NULL){
				free(SATCA_cnv_tbl);
				SATCA_cnv_tbl = NULL;
			}
			cnv_tbl_cnt = 0;
			tmcal_fsize = buf.st_size;
		}else{
#ifdef DBG_TITM
			printf("[%s:%d]Reading of a TIMEBASE_FILE is not needed.\n", __FILE__, __LINE__ );
#endif
			titm->SATCA_cnv_getTime = clt_time;
		}
	}
	/* Add Ver3.1.1 end   */	

	/*--------------------------------------------------------------*/
	/*   衛星時刻校正表から情報をテーブルにセット					*/
	/*--------------------------------------------------------------*/

/* Chg Ver3.1.1 start */
#if 0
	/* 指定されたファイル名のファイルをオープン */
	if((fp = fopen(titm->SATCA_Init.fname,"r")) == NULL){
#ifdef DBG_TITM
		printf("TiTm_Init[%d] TIMEBASE_FILE Open Error!! [%s] \n", __LINE__, fname );
#endif
		return(RET_FILE_OPEN_ERR);	/* Chg Ver2.0 */
	}
	/* ヘッダを読み込み衛星別パラメータをセット */
	if( (fgets( read_head, TBASE_REC_LENG+1, fp)) == NULL ) {
#ifdef DBG_TITM
		printf("TiTm_Init[%d] TIMEBASE_FILE Read Error!! [%s] \n", __LINE__, fname );
#endif
		fclose(fp);					/* Add Ver2.0*/

		/* Add Ver3.0.0 エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON4);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}

		return(RET_FILE_READ_ERR);	/* Chg Ver2.0 */
	}
	if(titm->SATCA_initial != 123454321){
		rtn = TiTmInt_headset( read_head, titm );
		if( rtn < 0 ) {
			fclose(fp);				/* Add Ver2.0*/
			return(rtn);			/* Chg Ver3.0.0 */
		}
	}
/* Chg Ver3.1.1 ----- */
#else
	if(titm->SATCA_initial != 123454321){
		cnv_tbl_cnt = TiTmInt_dataset( tmcal_fsize, &SATCA_cnv_tbl, titm );
		if( cnv_tbl_cnt < 0 ) {
			return(cnv_tbl_cnt);
		}
		titm->SATCA_cnv_getTime = clt_time;
		titm->SATCA_initial = 123454321;
	}
#endif
/* Chg Ver3.1.1 end   */

	if( inknd == 1 ) {
		/* Add&Chg Ver2.0 Start */
		/* inkindが1で検索範囲が0以外の場合、衛星時刻は検索開始の衛星時刻とする */
		SATstart = In_arg1;
		if( ((int)day != 0) || ((int)day2 != 0) ) {		/* Chg Ver3.0.0 add || ((int)day2 != 0) */
			SATstart -= (double)day * DAY_SEC;
		}

		titm->SATCA_Init.SATsec = SATstart;
/* Chg Ver3.1.1 start */
#if 0
		lpos = TiTmInt_fseek( inknd, fp, SATstart,  &last_lpos, titm );
#else
		rec_pos = TiTmInt_fseek( inknd, SATCA_cnv_tbl, SATstart,  cnv_tbl_cnt, titm );
#endif
/* Chg Ver3.1.1 end   */
		/* Add&Chg Ver2.0  End  */
	} else {
		In_ETI_TI = In_arg1 * (titm->SATCA_Init.xffmax+1.0) + In_arg2;	/* Chg Ver2.0 */
		titm->SATCA_Init.ETI_TI = In_ETI_TI;
/* Chg Ver3.1.1 start */
#if 0
		lpos = TiTmInt_fseek( inknd, fp, In_ETI_TI, &last_lpos, titm );
#else
		rec_pos = TiTmInt_fseek( inknd, SATCA_cnv_tbl, In_ETI_TI, cnv_tbl_cnt, titm );
#endif
/* Chg Ver3.1.1 end   */
	}
/* Del Ver3.1.1 start */
#if 0
	if( lpos < 0 ) {
		fclose(fp);		/* Add Ver2.0 */
		return(lpos);	/* Chg Ver3.0.0 */
	}

	rec_pos = lpos / titm->SATCA_cnv_rdlen;
#endif
/* Del Ver3.1.1 end   */
	/* Add Ver2.0 start */
	if( ((int)day != 0) || ((int)day2 != 0) ) {             /* Chg Ver3.0.0 add || ((int)day2 != 0) */
		top_pos = rec_pos - OVERLAP_READ_COUNT;
	}
	else {
	/* Add Ver2.0 end */
		top_pos = rec_pos - BEFORE_CNT;		/* Chg Ver2.0 */
	}
// 2012/02/17 LV0-001 ---start--- コメント行挿入のため先頭は3レコード目
//	if( top_pos <= 1 ) {
//		 top_pos = 1;
/* Chg Ver3.1.1 start */
#if 0
	if( top_pos <= DATA_TOP_RECORD ) {		/* Chg Ver2.0 */
		 top_pos = DATA_TOP_RECORD;			/* Chg Ver2.0 */
#else
	/* } */
	if( top_pos <= 0 ) {
		 top_pos = 0;
#endif
/* Chg Ver3.1.1 end   */
// 2012/02/17 LV0-001 ---end---
		/* 1:変換テーブルの先頭のレコードをメモリに展開した。*/
		titm->SATCA_cnv_topflg = 1;
	} else {
		titm->SATCA_cnv_topflg = 0;
	}
	/* Add Ver2.0 start */
	if( ((int)day != 0) || ((int)day2 != 0) ) {             /* Chg Ver3.0.0 add || ((int)day2 != 0) */
		last_pos =  rec_pos + BEFORE_CNT + OVERLAP_READ_COUNT;
	}
	else {
	/* Add Ver2.0 end */
		last_pos = rec_pos + AFTER_CNT;
	}
/* Chg Ver3.1.1 start */
#if 0
	if( last_pos >= (last_lpos/titm->SATCA_cnv_rdlen) ) {
		last_pos = last_lpos/titm->SATCA_cnv_rdlen;
#else
	/* } */
	if( last_pos >= (cnv_tbl_cnt - 1) ) {
		last_pos = cnv_tbl_cnt - 1;
#endif
/* Chg Ver3.1.1 end   */
		/* 1:変換テーブルの最後のレコードをメモリに展開した。*/
		titm->SATCA_cnv_endflg = 1;
	} else {
		titm->SATCA_cnv_endflg = 0;
	}

/* Chg Ver3.1.1 start */
#if 0
	lpos = top_pos * titm->SATCA_cnv_rdlen;
	fseek( fp, lpos, SEEK_SET );
	rec_pos = top_pos;
#ifdef DBG_TITM
	printf("TiTm_Init[%d] rec_pos[%d] top_pos[%d] last_pos[%d] lpos[%ld] \n", __LINE__, rec_pos, top_pos, last_pos, lpos );
#endif
#else
	rec_pos = top_pos;
#ifdef DBG_TITM
	printf("TiTm_Init[%d] rec_pos[%d] top_pos[%d] last_pos[%d] \n", __LINE__, rec_pos, top_pos, last_pos );
#endif
#endif
/* Chg Ver3.1.1 end   */

	/* 変換レコード数カウンタの初期化 */
	titm->SATCA_cnv_rdcnt = 0;
	titm->SATCA_cnv_errflg = 0;

/* Chg Ver3.1.1 start */
#if 0
	while(fgets( read_rd, TBASE_REC_LENG+1, fp) != NULL ){

		/* Add Ver3.0.0 エラーメッセージ出力 */
		if( strlen(read_rd) != titm->SATCA_cnv_rdlen ) {	/* Chg Ver3.0.0 "<" -> "!=" */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, rec_pos+1, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			fclose(fp);
			if(rtn != RET_NORMAL){
				return(rtn);
			}
			return(RET_FILE_READ_ERR);
		}

		/********************************************************/
		/* 検索範囲内のデータを抜出し、メモリ上に展開  			*/
		/********************************************************/
		TiTmInt_recdiv( read_rd, &rec_tbl, titm );	/* Chg Ver2.0 */

		/* Del Ver2.0 Start 衛星時間から抽出をE～Eまでとした場合、不連続レコードの連続等が処理できなくなるため。 */
		/* 衛星時間から抽出の場合はE～Eまでとする */
/*		if( inknd == 1 ) {
			if( rec_tbl.quality_flag[0] == 'E' ) {
				titm->SATCA_cnv_errflg = 1;
				if( In_arg1 >= rec_tbl.tpktTI_SATsec ){
					titm->SATCA_cnv_rdcnt = 0;
					if( rec_pos != 1 ) titm->SATCA_cnv_topflg = 0;
				} else if( In_arg1 < rec_tbl.tpktTI_SATsec ){
					if( rec_pos != 1 ) {
						if( rec_pos != (last_lpos/TBASE_REC_LENG) )  {
							titm->SATCA_cnv_endflg = 0;
						}
						break;
					}
				}
			}
		}*/
		/* Del Ver2.0  End  */

		memset(&titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt], '0', sizeof(titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt]));

		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktTI = rec_tbl.tpktTI;
		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktTI_SATsec = rec_tbl.tpktTI_SATsec;
		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktTI_dT = rec_tbl.tpktTI_dT;
		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktETITI = rec_tbl.tpktETITI;
		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].ETI = rec_tbl.ETI;
		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].quality_flag[0] = rec_tbl.quality_flag[0];
		titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].quality_flag[1] = rec_tbl.quality_flag[1];
#else
/* Chg Ver3.1.1 ----- */
	/* } */
	for(i = top_pos; i < last_pos + 1; i++) {
		memcpy(&titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt], &SATCA_cnv_tbl[i], sizeof(titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt]));
#endif
/* Chg Ver3.1.1 end   */

#ifdef DBG_TITM
		printf( "[%s:%d] :SATCA_cnv_rdcnt[%d] tpktTI[%lf] tpktTI_SATsec[%lf] tpktTI_dT[%lf] ETI[%d] quality_flag[%s]\n",
		__FILE__ , __LINE__, titm->SATCA_cnv_rdcnt, titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktTI, titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktTI_SATsec, titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].tpktTI_dT, titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].ETI, titm->SATCA_cnv_tbl[titm->SATCA_cnv_rdcnt].quality_flag) ;
#endif

		titm->SATCA_cnv_rdcnt++;
/* Del Ver3.1.1 start */
#if 0
		rec_pos++;
		if( rec_pos > last_pos ) break;
#endif
/* Del Ver3.1.1 end   */
		/* Del Ver2.0 Start */
		/* 衛星時間から抽出の場合はE～Eまでとする */
/*		if( inknd == 1 ) {
			if( rec_tbl.quality_flag[0] == 'E' ) {
				if( In_arg1 < rec_tbl.tpktTI_SATsec ){
					break;
				}
			}
		}*/
		/* Del Ver2.0  End  */
	}

/* Del Ver3.1.1 start */
#if 0
	/* 変換ファイルのクローズ */
	fclose(fp);
#endif
/* Del Ver3.1.1 end   */

	/****************************************************************/
	/*   復帰情報   						*/
	/****************************************************************/
#ifdef DBG_TITM
	printf( "[%s:%d] TiTm_Init  END [%d]\n", __FILE__ , __LINE__, titm->SATCA_cnv_rdcnt ) ;
#endif
	if( titm->SATCA_cnv_rdcnt == 0){
		/* 変換レコード数が無い場合 */
		/* 異常終了 */
#ifdef DBG_TITM
		printf("TiTm_Init[%d] Table Load Error!! [%s]\n", __LINE__, titm->SATCA_Init.fname );
#endif
		return(RET_FILE_READ_ERR);
	}

	/* 正常終了 */
#ifdef DBG_TITM
	printf("TiTm_Init[%d] Table Load Success. [%s] \n", __LINE__, titm->SATCA_Init.fname );
#endif

/* Chg Ver3.1.1 start */
#if 0
	time( &titm->SATCA_cnv_getTime );
	titm->SATCA_initial = 123454321;
	titm->SATCA_top_rec = top_pos - DATA_TOP_RECORD;	/* Add Ver2.0 */
#else
	titm->SATCA_top_rec = top_pos;
#endif
/* Chg Ver3.1.1 end   */

	return(RET_NORMAL);

}

/******************************************************************

	1レコードの内容を抽出する

	TiTmInt_recdiv( char *read_rd, SATCA_cnv *rec_tbl )

		in  char 	*read_rd  レコード領域
		out SATCA_cnv 	*rec_tbl  抽出領域

******************************************************************/
void TiTmInt_recdiv( char *read_rd, SATCA_cnv *rec_tbl, TiTm_tbl *titm )
{

	char *read_item;		/* 項目抜出し作業用 		*/
	char *cp_save;			/* 項目抜出し作業用 		*/
	int item_cnt;			/* トーカナイザーで抜出した項目数を数える 	*/
//	int i, rtn;		/* Del Ver2.0 */
//	double dwk;		/* Del Ver2.0 */
	char cwk_dtim[100];
//	char cwk[100];		/* Del Ver2.0 */
	char cwork[TBASE_REC_LENG+1];	/* Chg Ver3.0.0 500 -> 1023+1 */

#ifdef DBG_TITM
	printf( "[%s:%d] item[%s]\n", __FILE__ , __LINE__, read_rd ) ;
#endif
	/* 空白区切りで項目を抜出す */
	memset( cwork, 0x00, sizeof(cwork) );
	strcpy( cwork, read_rd );
#ifdef DBG_TITM
	printf( "[%s:%d] item[%s]\n", __FILE__ , __LINE__, cwork ) ;
#endif
	read_item = (char *)strtok_r(cwork, " \n", &cp_save);

	memset(rec_tbl, 0x00, sizeof(SATCA_cnv));	/* Chg Ver2.0 */
	item_cnt = 1;

	/* Chg Ver2.0 衛星時刻校正表の項目順変更に伴う処理順序変更など */
	/*--------------------------------------------------------------*/
	/* 1項目のETIを抜出す。  										*/
	/* 文字型からuint32_t型に型変換する。							*/
	/*--------------------------------------------------------------*/
	rec_tbl->ETI = (uint32_t)atoll( read_item );
#ifdef DBG_TITM
	printf( "[%s:%d] :ETI[%s][%u]   \n", __FILE__ , __LINE__, read_item, rec_tbl->ETI ) ;
#endif

	while((read_item = (char *)strtok_r(NULL, " \n", &cp_save)) != NULL){
		/* strtokで取得した項目数をカウントアップ */
		item_cnt++;
#ifdef DBG_TITM
		printf( "[%s:%d] cnt[%d] item[%s]\n", __FILE__ , __LINE__, item_cnt, read_item ) ;
#endif

		if(item_cnt == 2){
			/*--------------------------------------------------------------*/
			/* 2項目のTimePacketTIを抜出す。								*/
			/* 文字型からdouble型に型変換する。								*/
			/*--------------------------------------------------------------*/
			rec_tbl->tpktTI = atof( read_item );
			rec_tbl->tpktETITI = (double)rec_tbl->ETI * (titm->SATCA_Init.xffmax+1.0) + rec_tbl->tpktTI;
#ifdef DBG_TITM
			printf( "[%s:%d] :tpktTI[%s][%lf][%lf][%d][%lf]   \n", __FILE__ , __LINE__, read_item, rec_tbl->tpktTI, titm->SATCA_Init.xffmax, rec_tbl->ETI, rec_tbl->tpktETITI ) ;
#endif
		}else if(item_cnt == 3){
			/*--------------------------------------------------------------*/
			/* 3項目の時刻パケット 衛星時刻を抜出す  単位:秒				*/
			/* 文字型からdouble型に型変換する。								*/
			/*--------------------------------------------------------------*/
			memset( cwk_dtim, 0x00, sizeof(cwk_dtim) );
			strcpy( cwk_dtim, read_item );
			utc2es( TIME_STR_KIND_MMDD, cwk_dtim, &rec_tbl->tpktTI_SATsec );
#ifdef DBG_TITM
			printf( "[%s:%d] :tpktTI_SATsec[%s][%lf]   \n", __FILE__ , __LINE__, read_item, rec_tbl->tpktTI_SATsec ) ;
#endif
		}else if(item_cnt == 4){
			/*--------------------------------------------------------------*/
			/* 4項目のTimePacketTIのベース時刻を抜出す。  単位:秒			*/
			/* 文字型からdouble型に型変換する。								*/
			/*--------------------------------------------------------------*/
			rec_tbl->tpktTI_dT = atof(read_item) / (1.0E+6);
#ifdef DBG_TITM
			printf( "[%s:%d] :tpktTI_dT[%s][%lf]   \n", __FILE__ , __LINE__, read_item, rec_tbl->tpktTI_dT ) ;
#endif
		}else if( item_cnt == 5){
			/*--------------------------------------------------------------*/
			/* 5項目のクオリティーフラグを抜出す							*/
			/* 文字型のまま設定												*/
			/*--------------------------------------------------------------*/
			strcpy(rec_tbl->quality_flag, read_item);
			rec_tbl->quality_flag[1] = 0x00;
#ifdef DBG_TITM
			printf( "[%s:%d] :quality_flag[%s][%s]   \n", __FILE__ , __LINE__, read_item, rec_tbl->quality_flag ) ;
#endif
		}
	}
	return;		/* Chg Ver2.0 */
}

/******************************************************************

	ヘッダの内容を抽出する

	TiTmInt_headset( char *read_rd, TiTm_tbl *titm )

		in  char 	*read_rd  ヘッダ領域

1       2       3          4        5     6  7       8  9     10
#!HEAD: TIMESET 2011-02-09 10:04:03 01184 33 SOLAR-B 05 2.0.0 200

******************************************************************/
int TiTmInt_headset( char *read_rd, TiTm_tbl *titm )
{

	char *read_item;		/* 項目抜出し作業用 		*/
	char *cp_save;			/* 項目抜出し作業用 		*/
	int item_cnt;			/* トーカナイザーで抜出した項目数を数える 	*/
	int rtn;			/* Chg Ver3.0.0 */
	int data_cnt;		/* Add Ver3.1.1 */
//	char cwk[100];		/* Del Ver2.0 */
	char cwork[TBASE_REC_LENG+1];	/* Chg Ver3.0.0 300 -> 1023+1 */
	char errMsg[EMGS_MSG_LENGTH];	/* Add Ver3.0.0 エラーメッセージ本文 */

/* 空白区切りで項目を抜出す */
	memset( cwork, 0x00, sizeof(cwork) );
	strcpy( cwork, read_rd );
	read_item = (char *)strtok_r(cwork, " \n", &cp_save);

	item_cnt = 1;
	while((read_item = (char *)strtok_r(NULL, " \n", &cp_save)) != NULL){
		/* strtokで取得した項目数をカウントアップ */
		item_cnt++;

		if(item_cnt == 5){
			/*--------------------------------------------------------------*/
			/* 5項目のレコード数を抜き出す                        			*/
			/*--------------------------------------------------------------*/
			data_cnt = atoi(read_item);	/* Add Ver3.1.1 */
			if( data_cnt == 0 ){	/* Chg Ver3.1.1 atoi(read_item) -> data_cnt */
				sprintf(errMsg, EMSG_FILEREAD_ERR_FMT1, titm->SATCA_Init.fname, 1, 5, EMSG_FILEREAD_ERR_REASON2);
				rtn = outputMessage(errMsg);
				if(rtn != RET_NORMAL){
					return(rtn);
				}
				return(RET_FILE_READ_ERR);
			}
		/* Chg Ver2.0 衛星時刻校正表ヘッダレコードの項目順変更に伴う処理順序変更・削除など */
		}else if(item_cnt == 8){
			/*--------------------------------------------------------------*/
			/* 8項目のベース時刻を抜出す。                                 */
			/*--------------------------------------------------------------*/
			titm->SATCA_Init.SHeadTI_dT2n = atoi( read_item );
			titm->SATCA_Init.SHeadTI_dT = pow( 2.0, (double)(-(titm->SATCA_Init.SHeadTI_dT2n)) );
#ifdef DBG_TITM
			printf( "[%s:%d] :TPktTI_dT[%s][%d][%lf]   \n", __FILE__ , __LINE__, read_item, titm->SATCA_Init.SHeadTI_dT2n, titm->SATCA_Init.SHeadTI_dT ) ;
#endif
			titm->SATCA_Init.xffmax = (double)UINT32_MAX;
			titm->SATCA_Init.xffhalf = titm->SATCA_Init.xffmax / 2.0 ;
		}else if(item_cnt == 10){
			/* Add Ver3.0.0 */
			/*--------------------------------------------------------------*/
			/* 10項目のレコード長を抜出す。                                 */
			/*--------------------------------------------------------------*/
			titm->SATCA_cnv_rdlen = atoi(read_item);
			if(titm->SATCA_cnv_rdlen == 0){
				sprintf(errMsg, EMSG_FILEREAD_ERR_FMT1, titm->SATCA_Init.fname, 1, 10, EMSG_FILEREAD_ERR_REASON2);
				rtn = outputMessage(errMsg);
				if(rtn != RET_NORMAL){
					return(rtn);
				}
				return(RET_FILE_READ_ERR);
			}
		}
	}
	if( item_cnt < 9 ) {
		/* Add Ver3.0.0 エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT1, titm->SATCA_Init.fname, 1, 9, EMSG_FILEREAD_ERR_REASON1);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}

		return(RET_FILE_READ_ERR);		/* Chg Ver3.0.0 */
	}

	if(item_cnt == 9){		/* Add Ver3.0.0 レコード長が設定されていない場合は、固定値 */
		titm->SATCA_cnv_rdlen = TBASE_DEF_REC_LENG;
	}

	return(data_cnt);					/* Chg Ver3.0.0 *//* Chg Ver3.1.1 RET_NORMAL -> data_cnt */

}

/* Chg Ver3.1.1 start */
#if 0
/******************************************************************

	指定のレコードを求める。

	TiTmInt_fseek( int inknd, FILE *fp, double objval, long *last_lpos, TiTm_tbl *titm )

		in  int 	inknd  	検索種別
		in  FILE 	*fp  	ファイル識別子
		in  double 	objval  対象時刻(1:衛星時刻 2:ETI+TI)
		in  long 	*last_lpos 最終レコードポジション
		out long 	lpos   	ファイルポジション

******************************************************************/
long TiTmInt_fseek( int inknd, FILE *fp, double objval, long *last_lpos, TiTm_tbl *titm ) {

	int	rtn;
	char	rdata[TBASE_REC_LENG+1];	/* Chg Ver3.0.0 300 -> 1023+1 */
//	char	*rtnchar;		/* Del Ver2.0 */
	SATCA_cnv rec_tbl;
	long	lpos;
	int		pos, top_pos, last_pos;
	char errMsg[EMGS_MSG_LENGTH];	/* Add Ver3.0.0 エラーメッセージ本文 */
	int i;


	/* Chg Ver3.0.0 以降、全体的にレコード長変更 TBASE_REC_LENG -> titm->SATCA_cnv_rdlen */

								/* 最終レコードポジションを求める */
	lpos = (-1) * titm->SATCA_cnv_rdlen;
	fseek( fp, lpos, SEEK_END );
	lpos = ftell(fp);
// 2012/02/17 LV0-001 ---start--- コメント行挿入のため先頭は3レコード目
//	if( lpos < TBASE_REC_LENG ) { }
	if( lpos <= titm->SATCA_cnv_rdlen ) {
// 2012/02/17 LV0-001 ---end---
		lpos = RET_FILE_READ_ERR;	/* Chg Ver3.0.0 */
		/* Add Ver3.0.0 エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON5);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}
		return(lpos);
	}
	*last_lpos = lpos;			/* 処理位置変更 Ver2.0*/

								/* 先頭レコードとの比較を行う */
// 2012/02/17 LV0-001 ---start--- コメント行挿入のため先頭は3レコード目
//	lpos = TBASE_REC_LENG;
//	lpos = titm->SATCA_cnv_rdlen + titm->SATCA_cnv_rdlen;	/* Del Ver3.0.0 */
// 2012/02/17 LV0-001 ---end---
	fseek( fp, 0, SEEK_SET );		/* Chg Ver3.0.0 lops -> 0 */

	for( i=0; i<3; i++ ){
		if( (fgets( rdata, TBASE_REC_LENG+1, fp )) == NULL ) {
			lpos = RET_FILE_READ_ERR;	/* Chg Ver3.0.0 */
			/* Add Ver3.0.0 エラーメッセージ出力 */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, i+1, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			if(rtn != RET_NORMAL){
				return(rtn);
			}
			return(lpos);
		}
		if( strlen(rdata) != titm->SATCA_cnv_rdlen ) {		/* Chg Ver3.0.0 "<" -> "!=" */
			lpos = RET_FILE_READ_ERR;	/* Chg Ver3.0.0 */
			/* Add Ver3.0.0 エラーメッセージ出力 */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, i+1, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			if(rtn != RET_NORMAL){
				return(rtn);
			}
			return(lpos);
		}
	}

#ifdef DBG_TITM
	printf( "TOP[%ld][%d]\n", lpos, rtn );
#endif

	TiTmInt_recdiv( rdata, &rec_tbl, titm );	/* Chg Ver2.0 */
								/* 先頭より前ならば先頭として復帰 */
	if( inknd == 1 ) {
		if( objval <= rec_tbl.tpktTI_SATsec ) {
			lpos = ftell(fp) - titm->SATCA_cnv_rdlen;
			fseek( fp, lpos, SEEK_SET );
#ifdef DBG_TITM
			printf( "top only[%ld]\n", lpos );
#endif
			return(lpos);
		}
	} else {
		if( objval <= rec_tbl.tpktETITI ) {
			lpos = ftell(fp) - titm->SATCA_cnv_rdlen;
			fseek( fp, lpos, SEEK_SET );
#ifdef DBG_TITM
			printf( "top only[%ld]\n", lpos );
#endif
			return(lpos);
		}
	}

	/* 最終レコードとの比較を行う */
	lpos = (-1) * titm->SATCA_cnv_rdlen;
	fseek( fp, lpos, SEEK_END );
	lpos = ftell(fp);
	if( lpos <= titm->SATCA_cnv_rdlen ) {		/* Add Ver2.0*/
		/* Add Ver3.0.0 エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON5);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}
		return(RET_FILE_READ_ERR);	/* Chg Ver3.0.0 */
	}
	last_pos = lpos / titm->SATCA_cnv_rdlen;

	fgets( rdata, TBASE_REC_LENG+1, fp );

	/* Add Ver3.0.0 エラーメッセージ出力 */
	if( strlen(rdata) != titm->SATCA_cnv_rdlen ) {		/* Chg Ver3.0.0 "<" -> "!=" */
		lpos = RET_FILE_READ_ERR;	/* Chg Ver3.0.0 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON5);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}

		return(lpos);
	}

#ifdef DBG_TITM
	printf( "LAST[%ld][%d]\n", lpos, rtn );
#endif

	TiTmInt_recdiv( rdata, &rec_tbl, titm );	/* Chg Ver2.0 */
	/* 最終より後ならば最終として復帰 */
	if( inknd == 1 ) {
		if( objval >= rec_tbl.tpktTI_SATsec ) {
			lpos = ftell(fp) - titm->SATCA_cnv_rdlen;
			fseek( fp, lpos, SEEK_SET );
#ifdef DBG_TITM
			printf( "last only[%ld]\n", lpos );
#endif
			return(lpos);
		}
	} else {
		if( objval >= rec_tbl.tpktETITI ) {
			lpos = ftell(fp) - titm->SATCA_cnv_rdlen;
			fseek( fp, lpos, SEEK_SET );
#ifdef DBG_TITM
			printf( "last only[%ld]\n", lpos );
#endif
			return(lpos);
		}
	}

								/* 二分割で目的のレコードを求める */
// 2012/02/17 LV0-001 ---start--- コメント行挿入のため先頭は3レコード目
//	top_pos = 1;
	top_pos = DATA_TOP_RECORD;		/* Chg Ver2.0 */
// 2012/02/17 LV0-001 ---end---
#ifdef DBG_TITM
	printf( "seek start[%d][%d]\n", top_pos, last_pos );
#endif
	for(;;) {
		if( (top_pos == last_pos) || ((top_pos+1) == last_pos) ) {	/* 1レコードまたは、2レコードしかない場合 */
			break;
		}
		pos = top_pos + (last_pos-top_pos) / 2;
		lpos = pos * titm->SATCA_cnv_rdlen;
#ifdef DBG_TITM
		printf( "seek[%d][%d][%d][%ld]\n", pos,top_pos,last_pos,lpos );
#endif
		fseek( fp, lpos, SEEK_SET );
		fgets( rdata, TBASE_REC_LENG+1, fp );
		/* Add Ver3.0.0 エラーメッセージ出力 */
		if( strlen(rdata) != titm->SATCA_cnv_rdlen ) {		/* Chg Ver3.0.0 "<" -> "!=" */
			lpos = RET_FILE_READ_ERR;	/* Chg Ver3.0.0 */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, pos+1, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			if(rtn != RET_NORMAL){
				return(rtn);
			}

			return(lpos);
		}

		TiTmInt_recdiv( rdata, &rec_tbl, titm );	/* Chg Ver2.0 */
		if( inknd == 1 ) {
			if( objval == rec_tbl.tpktTI_SATsec ) {
				top_pos = pos;
				break;
			} else if( objval < rec_tbl.tpktTI_SATsec ) {
				last_pos = pos;
			} else {
				top_pos = pos;
			}
		} else {
			if( objval == rec_tbl.tpktETITI ) {
				last_pos = pos;
				break;
			} else if( objval < rec_tbl.tpktETITI ) {
				last_pos = pos;
			} else {
				top_pos = pos;
			}
		}
	}

								/* 目的のレコードを情報としてセット */
	lpos = top_pos * titm->SATCA_cnv_rdlen;
	fseek( fp, lpos, SEEK_SET );
	lpos = ftell(fp);
#ifdef DBG_TITM
	printf( "seek end[%ld][%ld]\n", top_pos,lpos );
#endif

	return(lpos);
}

/* Chg Ver3.1.1 ----- */
#else
/******************************************************************

	指定のレコードを求める。

	TiTmInt_fseek( int inknd, SATCA_cnv *SATCA_cnv_tbl, double objval, int tbl_cnt, TiTm_tbl *titm )

		in  int 	inknd  	検索種別
		in  SATCA_cnv 	*SATCA_cnv_tbl  	メモリ領域
		in  double 	objval  対象時刻(1:衛星時刻 2:ETI+TI)
		in  int 	tbl_cnt メモリ領域テーブル数
		out int 	lpos   	メモリ領域ポジション

******************************************************************/
int TiTmInt_fseek( int inknd, SATCA_cnv *SATCA_cnv_tbl, double objval, int tbl_cnt, TiTm_tbl *titm ) {

	int pos, top_pos, last_pos;

	last_pos = tbl_cnt-1;
	if( inknd == 1 ) {
		/* 先頭より前ならば先頭として復帰 */
		if( objval <= SATCA_cnv_tbl[0].tpktTI_SATsec ) {
#ifdef DBG_TITM
			printf( "top only[%ld]\n", 0 );
#endif
			return(0);
		}
		/* 最終より後ならば最終として復帰 */
		if( objval >= SATCA_cnv_tbl[last_pos].tpktTI_SATsec ) {
#ifdef DBG_TITM
			printf( "last only[%ld]\n", last_pos );
#endif
			return(last_pos);
		}
	} else {
		/* 先頭より前ならば先頭として復帰 */
		if( objval <= SATCA_cnv_tbl[0].tpktETITI ) {
#ifdef DBG_TITM
			printf( "top only[%ld]\n", 0 );
#endif
			return(0);
		}
		/* 最終より後ならば最終として復帰 */
		if( objval >= SATCA_cnv_tbl[last_pos].tpktETITI ) {
#ifdef DBG_TITM
			printf( "last only[%ld]\n", last_pos );
#endif
			return(last_pos);
		}
	}
	/* 二分割で目的のレコードを求める */
	top_pos = 0;
#ifdef DBG_TITM
	printf( "seek start[%d][%d]\n", top_pos, last_pos );
#endif
	for(;;) {
		/* 1レコードまたは、2レコードしかない場合 */
		if( (top_pos == last_pos) || ((top_pos+1) == last_pos) ) {
			break;
		}
		pos = top_pos + (last_pos-top_pos) / 2;
		if( inknd == 1 ) {
			if( objval == SATCA_cnv_tbl[pos].tpktTI_SATsec ) {
				top_pos = pos;
				break;
			} else if( objval < SATCA_cnv_tbl[pos].tpktTI_SATsec ) {
				last_pos = pos;
			} else {
				top_pos = pos;
			}
		} else {
			if( objval == SATCA_cnv_tbl[pos].tpktETITI ) {
				last_pos = pos;
				break;
			} else if( objval < SATCA_cnv_tbl[pos].tpktETITI ) {
				last_pos = pos;
			} else {
				top_pos = pos;
			}
		}
	}
#ifdef DBG_TITM
	printf( "seek end[%ld][%ld]\n", top_pos, tbl_cnt-1 );
#endif

	return(top_pos);
}
#endif
/* Chg Ver3.1.1 end   */

/* Add Ver3.1.1 start */
/******************************************************************

	衛星時刻校正表のデータレコードをメモリ領域に展開する。

	TiTmInt_dataset( int tmcal_fsize, SATCA_cnv **SATCA_cnv_tbl, TiTm_tbl *titm )

		in  int       tmcal_fsize		衛星時刻校正表ファイルサイズ
		out SATCA_cnv **SATCA_cnv_tbl   メモリ領域	

******************************************************************/
int TiTmInt_dataset( int tmcal_fsize, SATCA_cnv **SATCA_cnv_tbl, TiTm_tbl *titm ) {

	FILE *fp;
	int i, rtn;
	int tbl_cnt;
	SATCA_cnv *SATCA_cnv_tbl_wk;
	char rdata[TBASE_REC_LENG+1];	/* 校正表１レコード分を格納 */
	char errMsg[EMGS_MSG_LENGTH];	/* エラーメッセージ */

	/* 指定されたファイル名のファイルをオープン */
	if((fp = fopen(titm->SATCA_Init.fname, "r")) == NULL){
#ifdef DBG_TITM
		printf("TiTmInt_dataset[%d] TIMEBASE_FILE Open Error!! [%s] \n", __LINE__, titm->SATCA_Init.fname );
#endif
		return(RET_FILE_OPEN_ERR);
	}
	/* ヘッダを読み込み衛星別パラメータをセット */
	if( (fgets( rdata, TBASE_REC_LENG+1, fp)) == NULL ) {
#ifdef DBG_TITM
		printf("TiTmInt_dataset[%d] TIMEBASE_FILE Read Error!! [%s] \n", __LINE__, titm->SATCA_Init.fname );
#endif
		fclose(fp);

		/* エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON4);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}
		return(RET_FILE_READ_ERR);
	}
	tbl_cnt = TiTmInt_headset( rdata, titm );
	if( tbl_cnt < 0 ) {
		fclose(fp);
		return(tbl_cnt);
	}
	/* データレコードカウントチェック */
	if( tmcal_fsize != titm->SATCA_cnv_rdlen * (tbl_cnt + 2) ) {
#ifdef DBG_TITM
		printf("TiTmInt_dataset[%d] TIMEBASE_FILE Record count unmatch!! [%s] \n", __LINE__, titm->SATCA_Init.fname );
#endif
		fclose(fp);

		/* エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON6);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}
		return(RET_FILE_READ_ERR);
	}

	/* 先頭レコードとの比較を行う */
	fseek( fp, 0, SEEK_SET );
	for( i = 0; i < 3; i++ ) {
		if( (fgets( rdata, TBASE_REC_LENG+1, fp )) == NULL ) {
			fclose(fp);
			/* エラーメッセージ出力 */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, i+1, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			if(rtn != RET_NORMAL){
				return(rtn);
			}
			return(RET_FILE_READ_ERR);
		}
		if( strlen(rdata) != titm->SATCA_cnv_rdlen ) {
			fclose(fp);
			/* エラーメッセージ出力 */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, i+1, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			if(rtn != RET_NORMAL){
				return(rtn);
			}
			return(RET_FILE_READ_ERR);
		}
	}
	/* メモリ領域獲得 */
	SATCA_cnv_tbl_wk = (SATCA_cnv *)malloc(sizeof(SATCA_cnv)*tbl_cnt);
	if( SATCA_cnv_tbl_wk == NULL ) {
#ifdef DBG_TITM
		printf("TiTmInt_dataset[%d] Memory Allocate Error!! [%s] \n", __LINE__, titm->SATCA_Init.fname );
#endif
		fclose(fp);
		/* エラーメッセージ出力 */
		sprintf(errMsg, EMSG_FILEREAD_ERR_FMT3, titm->SATCA_Init.fname, EMSG_FILEREAD_ERR_REASON7);
		rtn = outputMessage(errMsg);
		if(rtn != RET_NORMAL){
			return(rtn);
		}
		return(RET_FILE_READ_ERR);
	}
	/* メモリ領域にデータレコード情報設定 */
	TiTmInt_recdiv( rdata, &SATCA_cnv_tbl_wk[0], titm );
	i = 1;
	while(fgets( rdata, TBASE_REC_LENG+1, fp) != NULL ){
		if( strlen(rdata) != titm->SATCA_cnv_rdlen ) {
			fclose(fp);
			free(SATCA_cnv_tbl_wk);
			/* エラーメッセージ出力 */
			sprintf(errMsg, EMSG_FILEREAD_ERR_FMT2, titm->SATCA_Init.fname, i+3, EMSG_FILEREAD_ERR_REASON3);
			rtn = outputMessage(errMsg);
			if(rtn != RET_NORMAL){
				return(rtn);
			}
			return(RET_FILE_READ_ERR);
		}
		TiTmInt_recdiv( rdata, &SATCA_cnv_tbl_wk[i], titm );
		i++;
	}
	fclose(fp);

	*SATCA_cnv_tbl = SATCA_cnv_tbl_wk;

	return(tbl_cnt);
}
/* Add Ver3.1.1 end   */

/******************************************************************

	メッセージ出力

	outputMessage( char *msg )

		in  char 	msg  	出力するメッセージ本文

******************************************************************/
int outputMessage( char *msg ){

	FILE *fp;
	char msgFilePath[FILE_PATH_MAX_LENGTH];

	/* Ver3.0.1 add */
	struct  tm  *tm;
	time_t  tim;
	unsigned int  inen;
	unsigned char  msgHead[50];

	filepath_mng(ERROR_FILE_PATH_GET, msgFilePath);

	/* Add Ver3.1.0 Str */
	if ( msg == NULL ) {
		return(RET_FILE_OPEN_ERR);
	}
	/* Add Ver3.1.0 End */

	if((fp = fopen(msgFilePath,"a")) == NULL){
			return(RET_FILE_OPEN_ERR);
	}
	
	/* Ver3.0.1 add start */
	time( &tim );
	tm = localtime( &tim );
	if ( tm->tm_year <= 50 ) {
		inen = 2000 + tm->tm_year;
	} else {
		inen = 1900 + tm->tm_year;
	}

	memset( msgHead, 0x00, 50 );
	sprintf( ( char * )msgHead, EMSG_FILEREAD_ERR_FMT0
		, inen
		, ( tm->tm_mon + 1 )
		, tm->tm_mday
		, tm->tm_hour
		, tm->tm_min
		, tm->tm_sec
	);

	fputs(msgHead, fp);
	/* Ver3.0.1 add end */

	fputs(msg, fp);

	fclose(fp);

	return(RET_NORMAL);

}
